/*
 * IPVS         An implementation of the IP virtual server support for the
 *              LINUX operating system.  IPVS is now implemented as a module
 *              over the Netfilter framework. IPVS can be used to build a
 *              high-performance and highly available server based on a
 *              cluster of servers.
 *
 * Version:     $Id: ip_vs_core.c,v 1.12 2001/01/26 10:19:23 wensong Exp $
 *
 * Authors:     Wensong Zhang <wensong@linuxvirtualserver.org>
 *              Peter Kese <peter.kese@ijs.si>
 *              Julian Anastasov <uli@linux.tu-varna.acad.bg>
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *
 * The IPVS code for kernel 2.2 was done by Wensong Zhang and Peter Kese,
 * with changes/fixes from Julian Anastasov, Lars Marowsky-Bree, Horms
 * and others.
 *
 * Changes:
 *
 */

#ifdef MODULE
#define EXPORT_SYMTAB
#endif

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/icmp.h>

#include <net/ip.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <net/icmp.h>                   /* for icmp_send */
#include <net/route.h>

#include <linux/netfilter.h>
#include <linux/netfilter_ipv4.h>

#include "ip_vs.h"
//#include <net/ip_vs.h>


EXPORT_SYMBOL(register_ip_vs_scheduler);
EXPORT_SYMBOL(unregister_ip_vs_scheduler);
EXPORT_SYMBOL(ip_vs_bind_dest);
EXPORT_SYMBOL(ip_vs_unbind_dest);
EXPORT_SYMBOL(ip_vs_skb_replace);
EXPORT_SYMBOL(ip_vs_proto_name);
EXPORT_SYMBOL(ip_vs_conn_new);
EXPORT_SYMBOL(ip_vs_conn_in_get);
EXPORT_SYMBOL(ip_vs_conn_out_get);
EXPORT_SYMBOL(ip_vs_conn_listen);
EXPORT_SYMBOL(ip_vs_conn_put);
#ifdef CONFIG_IP_VS_DEBUG
EXPORT_SYMBOL(ip_vs_get_debug_level);
#endif
EXPORT_SYMBOL(check_for_ip_vs_out);


/* ID used in ICMP lookups */
#define icmp_id(icmph)          ((icmph->un).echo.id)


/*
 *	Get transport protocol data offset, check against size
 *	return:
 *		0  if other IP proto
 *		-1 if error
 */
int ip_vs_proto_doff(unsigned proto, char *th, unsigned size)
{
	int ret = -1;
	switch (proto) {
        case IPPROTO_ICMP:
                if (size >= sizeof(struct icmphdr))
                        ret = sizeof(struct icmphdr);
                break;
        case IPPROTO_UDP:
                if (size >= sizeof(struct udphdr))
                        ret = sizeof(struct udphdr);
                break;
        case IPPROTO_TCP:
                /*
                 *	Is this case, this check _also_ avoids
                 *	touching an invalid pointer if
                 *	size is invalid
                 */
                if (size >= sizeof(struct tcphdr)) {
                        ret = ((struct tcphdr*)th)->doff << 2;
                        if (ret > size) {
                                ret = -1 ;
                        }
                }
                break;
        default:
                /* Other proto: nothing to say, by now :) */
                ret = 0;
	}
	if (ret < 0)
		IP_VS_DBG(0, "mess proto_doff for proto=%d, size =%d\n",
                          proto, size);
	return ret;
}


const char *ip_vs_proto_name(unsigned proto)
{
	static char buf[20];

        switch (proto) {
        case IPPROTO_IP:
                return "IP";
        case IPPROTO_UDP:
                return "UDP";
        case IPPROTO_TCP:
                return "TCP";
        case IPPROTO_ICMP:
                return "ICMP";
        default:
		sprintf(buf, "IP_%d", proto);
		return buf;
        }
}


/*
 *	Own skb_cow() beast, tweaked for rewriting commonly
 *	used pointers in the code
 */
static struct sk_buff *vs_skb_cow(struct sk_buff **skb_p,
                                  struct iphdr **iph_p, unsigned char **t_p)
{
	struct sk_buff *skb=(*skb_p);
	if (skb_cloned(skb)) {
		skb = skb_copy(skb, GFP_ATOMIC);
		if (skb) {
			/*
			 *	skb changed, update other pointers
			 */
			struct iphdr *iph = skb->nh.iph;
			kfree_skb(*skb_p);
			*skb_p = skb;
			*iph_p = iph;
			*t_p = (char*) iph + iph->ihl * 4;
		}
	}
	return skb;
}


/*
 *  IPVS persistent scheduling function
 *  It creates a connection entry according to its template if exists,
 *  or selects a server and creates a connection entry plus a template.
 *  Locking: we are svc user (svc->refcnt), so we hold all dests too
 */
static struct ip_vs_conn *
ip_vs_sched_persist(struct ip_vs_service *svc, struct iphdr *iph)
{
	struct ip_vs_conn *cp = NULL;
	struct ip_vs_dest *dest;
	const __u16 *portp;
	struct ip_vs_conn *ct;
	__u16  dport;	 /* destination port to forward */
	__u32  snet;	 /* source network of the client, after masking */

	portp = (__u16 *)&(((char *)iph)[iph->ihl*4]);

	/* Mask saddr with the netmask to adjust template granularity */
	snet = iph->saddr & svc->netmask;

	IP_VS_DBG(6, "P-schedule: src %u.%u.%u.%u:%u dest %u.%u.%u.%u:%u "
                  "mnet %u.%u.%u.%u\n",
		  NIPQUAD(iph->saddr), ntohs(portp[0]),
		  NIPQUAD(iph->daddr), ntohs(portp[1]),
		  NIPQUAD(snet));

	/*
	 * As far as we know, FTP is a very complicated network protocol, and
	 * it uses control connection and data connections. For active FTP,
	 * FTP server initilize data connection to the client, its source port
	 * is often 20. For passive FTP, FTP server tells the clients the port
	 * that it passively listens to,  and the client issues the data
	 * connection. In the tunneling or direct routing mode, the load
	 * balancer is on the client-to-server half of connection, the port
	 * number is unknown to the load balancer. So, a conn template like
	 * <caddr, 0, vaddr, 0, daddr, 0> is created for persistent FTP
	 * service, and a template like <caddr, 0, vaddr, vport, daddr, dport>
	 * is created for other persistent services.
	 */
	if (portp[1] == svc->port) {
		/* Check if a template already exists */
		if (svc->port != FTPPORT)
			ct = ip_vs_conn_in_get(iph->protocol, snet, 0,
					       iph->daddr, portp[1]);
		else
			ct = ip_vs_conn_in_get(iph->protocol, snet, 0,
					       iph->daddr, 0);

		if (!ct || !ip_vs_check_template(ct)) {
			/*
			 * No template found or the dest of the connection
			 * template is not available.
			 */
			dest = svc->scheduler->schedule(svc, iph);
			if (dest == NULL) {
				IP_VS_DBG(1, "P-schedule: no dest found.\n");
				return NULL;
			}

                        /*
                         * Create a template like <protocol,daddr,0,
                         * maddr,mport,saddr,sport> for non-ftp service,
                         * and <protocol,daddr,0,maddr,0,saddr,0>
                         * for ftp service.
                         */
			if (svc->port != FTPPORT)
				ct = ip_vs_conn_new(iph->protocol,
						    snet, 0,
						    iph->daddr, portp[1],
						    dest->addr, dest->port,
						    0);
			else
				ct = ip_vs_conn_new(iph->protocol,
						    snet, 0,
						    iph->daddr, 0,
						    dest->addr, 0,
						    0);
			if (ct == NULL) {
				IP_VS_ERR("ip_vs_conn_new template failed\n");
				return NULL;
			}

			/*
			 * Bind the template with dest and set timeout.
			 */
			ip_vs_bind_dest(ct, dest);
			ct->timeout = svc->timeout;
		} else {
			/*
			 * Template found and its destination is available.
			 */
			dest = ct->dest;

			/*
			 * Delete its timer so that it can be put back.
			 */
			del_sltimer(&ct->timer);
		}
		dport = dest->port;
	} else {
                /*
                 * Note: persistent fwmark-based services and persistent
                 * port zero service are handled here.
                 * fwmark template: <IPPROTO_IP,daddr,0,maddr,0,saddr,0>
                 * port zero template: <protocol,daddr,0,maddr,0,saddr,0>
                 */
                if (svc->fwmark)
                        ct = ip_vs_conn_in_get(IPPROTO_IP, snet, 0,
                                               iph->daddr, 0);
                else
                        ct = ip_vs_conn_in_get(iph->protocol, snet, 0,
                                               iph->daddr, 0);

		if (!ct || !ip_vs_check_template(ct)) {
			/*
			 * If it is not persistent port zero, return NULL,
			 * otherwise create a connection template.
			 */
			if (svc->port)
				return NULL;

			dest = svc->scheduler->schedule(svc, iph);
			if (dest == NULL) {
				IP_VS_DBG(1, "P-schedule: no dest found.\n");
				return NULL;
			}

                        /*
                         * Create a template according to the service
                         */
                        if (svc->fwmark)
                                ct = ip_vs_conn_new(IPPROTO_IP,
                                                    snet, 0,
                                                    iph->daddr, 0,
                                                    dest->addr, 0,
                                                    0);
                        else
                                ct = ip_vs_conn_new(iph->protocol,
                                                    snet, 0,
                                                    iph->daddr, 0,
                                                    dest->addr, 0,
                                                    0);
			if (ct == NULL) {
				IP_VS_ERR("ip_vs_conn_new template failed\n");
				return NULL;
			}

			/*
			 * Bind the template with dest and set timeout.
			 */
			ip_vs_bind_dest(ct, dest);
			ct->timeout = svc->timeout;
		} else {
			dest = ct->dest;

			/*
			 * Delete its timer so that it can be put back.
			 */
			del_sltimer(&ct->timer);
		}
		dport = portp[1];
	}

	/*
	 *    Create a new connection according to the template
	 */
	cp = ip_vs_conn_new(iph->protocol,
			    iph->saddr, portp[0],
			    iph->daddr, portp[1],
			    dest->addr, dport,
			    0);
	if (cp == NULL) {
		IP_VS_ERR("ip_vs_conn_new failed\n");
		ip_vs_conn_put(ct);
		return NULL;
	}

	/*
	 *    Bind the connection entry with the vs dest.
	 */
	ip_vs_bind_dest(cp, dest);

	/*
	 *    Increase the inactive connection counter
	 *    because it is in Syn-Received
	 *    state (inactive) when the connection is created.
	 */
	atomic_inc(&dest->inactconns);

	/*
	 *    Add its control
	 */
	ip_vs_control_add(cp, ct);

	ip_vs_conn_put(ct);
	return cp;
}


/*
 *  IPVS main scheduling function
 *  It selects a server according to the virtual service, and
 *  creates a connection entry.
 */
struct ip_vs_conn *ip_vs_schedule(struct ip_vs_service *svc, struct iphdr *iph)
{
	struct ip_vs_conn *cp = NULL;
	struct ip_vs_dest *dest;
	const __u16 *portp;

	/*
	 *    Persistent service
	 */
	if (svc->flags & IP_VS_SVC_F_PERSISTENT)
		return ip_vs_sched_persist(svc, iph);

	/*
	 *    Non-persistent service
	 */
	portp = (__u16 *)&(((char *)iph)[iph->ihl*4]);
	if (!svc->fwmark && portp[1] != svc->port) {
		if (!svc->port)
			IP_VS_ERR("Schedule: port zero only supported "
                                  "in persistent services, "
                                  "check your ipvs configuration\n");
		return NULL;
	}

	dest = svc->scheduler->schedule(svc, iph);
	if (dest == NULL) {
		IP_VS_DBG(1, "Schedule: no dest found.\n");
		return NULL;
	}

	/*
	 *    Create a connection entry.
	 */
	cp = ip_vs_conn_new(iph->protocol,
			    iph->saddr, portp[0],
			    iph->daddr, portp[1],
			    dest->addr, dest->port?dest->port:portp[1],
			    0);
	if (cp == NULL) {
		IP_VS_ERR("Schedule: ip_vs_conn_new failed\n");
		return NULL;
	}

	/*
	 *    Bind the connection entry with the vs dest.
	 */
	ip_vs_bind_dest(cp, dest);

	/*
	 *    Increase the inactive connection counter because it is in
	 *    Syn-Received state (inactive) when the connection is created.
	 */
	atomic_inc(&dest->inactconns);

	IP_VS_DBG(6, "Schedule fwd:%c s:%s c:%u.%u.%u.%u:%u v:%u.%u.%u.%u:%u "
		  "d:%u.%u.%u.%u:%u flg:%X cnt:%d\n",
		  ip_vs_fwd_tag(cp), ip_vs_state_name(cp->state),
		  NIPQUAD(cp->caddr), ntohs(cp->cport),
		  NIPQUAD(cp->vaddr), ntohs(cp->vport),
		  NIPQUAD(cp->daddr), ntohs(cp->dport),
		  cp->flags, atomic_read(&cp->refcnt));

	return cp;
}


/*
 *  Pass or drop the packet.
 *  Called by ip_vs_in, when the virtual service is available but
 *  no destination is available for a new connection.
 */
static int ip_vs_leave(struct ip_vs_service *svc, struct sk_buff *skb)
{
	struct iphdr *iph = skb->nh.iph;
	__u16 *portp = (__u16 *)&(((char *)iph)[iph->ihl*4]);

        /* if it is fwmark-based service, the cache_bypass sysctl is up
           and the destination is RTN_UNICAST (and not local), then create
           a cache_bypass connection entry */
        if (sysctl_ip_vs_cache_bypass && svc->fwmark
            && (inet_addr_type(iph->daddr) == RTN_UNICAST)) {
                int ret;
                struct ip_vs_conn *cp;
                
                ip_vs_service_put(svc);

                /* create a new connection entry */
                IP_VS_DBG(6, "ip_vs_leave: create a cache_bypass entry\n");
                cp = ip_vs_conn_new(iph->protocol,
                                    iph->saddr, portp[0],
                                    iph->daddr, portp[1],
                                    0, 0, 0);
                if (cp == NULL) {
                        IP_VS_ERR("ip_vs_conn_new failed\n");
                        kfree_skb(skb);
                        return NF_STOLEN;
                }

                /* bind the bypass_xmit */
                ip_vs_bind_bypass_xmit(cp);

                /* set state */
                ip_vs_set_state(cp, VS_STATE_INPUT, iph, portp); 

                /* transmit the first SYN packet */
                ret = cp->packet_xmit(skb, cp);
                
                ip_vs_conn_put(cp);
                return ret;
        }

        /*
	 * When the virtual ftp service is presented, packets destined
	 * for other services on the VIP may get here (except services
	 * listed in the ipvs table), pass the packets, because it is
	 * not ipvs job to decide to drop the packets.
	 */
	if ((svc->port == FTPPORT) && (portp[1] != FTPPORT)) {
                ip_vs_service_put(svc);
		return NF_ACCEPT;
	}

        ip_vs_service_put(svc);

	/*
	 * Notify the client that the destination is unreachable, and
	 * release the socket buffer.
	 * Since it is in IP layer, the TCP socket is not actually
	 * created, the TCP RST packet cannot be sent, instead that
	 * ICMP_PORT_UNREACH is sent here no matter it is TCP/UDP. --WZ
	 */
	icmp_send(skb, ICMP_DEST_UNREACH, ICMP_PORT_UNREACH, 0);
	kfree_skb(skb);
	return NF_STOLEN;
}


static unsigned int ip_vs_post_routing(unsigned int hooknum,
                                       struct sk_buff **skb_p,
                                       const struct net_device *in,
                                       const struct net_device *out,
                                       int (*okfn)(struct sk_buff *))
{
        struct sk_buff  *skb = *skb_p;

	if (skb->nfmark != NFMARK_IPVS_PROPERTY)
                return NF_ACCEPT;

	/* The packet was sent from IPVS, exit this chain */
        (*okfn)(skb);
        
	return NF_STOLEN;
}

/*
 *	Handle ICMP messages in the inside-to-outside direction (outgoing).
 *	Find any that might be relevant, check against existing connections,
 *	forward to the right destination host if relevant.
 *	Currently handles error types - unreachable, quench, ttl exceeded.
 *      (Only used in VS/NAT)
 */
static int ip_vs_out_icmp(struct sk_buff **skb_p)
{
	struct sk_buff	*skb   = *skb_p;
	struct iphdr	*iph   = skb->nh.iph;
	struct icmphdr	*icmph = (struct icmphdr *)((char *)iph + (iph->ihl<<2));
	struct iphdr	*ciph;	/* The ip header contained within the ICMP */
	__u16		*pptr;	/* port numbers from TCP/UDP contained header */
	struct ip_vs_conn *cp;
	unsigned short	 len   = ntohs(iph->tot_len) - (iph->ihl * 4);

	IP_VS_DBG(12, "outgoing ICMP (%d,%d) %u.%u.%u.%u->%u.%u.%u.%u\n",
		  icmph->type, ntohs(icmp_id(icmph)),
		  NIPQUAD(iph->saddr), NIPQUAD(iph->daddr));

	/*
	 * Work through seeing if this is for us.
	 * These checks are supposed to be in an order that
	 * means easy things are checked first to speed up
	 * processing.... however this means that some
	 * packets will manage to get a long way down this
	 * stack and then be rejected, but thats life
	 */
	if ((icmph->type != ICMP_DEST_UNREACH) &&
	    (icmph->type != ICMP_SOURCE_QUENCH) &&
	    (icmph->type != ICMP_TIME_EXCEEDED))
		return NF_ACCEPT;

	/* Now find the contained IP header */
	ciph = (struct iphdr *) (icmph + 1);

	/* We are only interested ICMPs generated from TCP or UDP packets */
	if ((ciph->protocol != IPPROTO_UDP) && (ciph->protocol != IPPROTO_TCP))
		return NF_ACCEPT;

	/*
	 * Find the ports involved - this packet was
	 * incoming so the ports are right way round
	 * (but reversed relative to outer IP header!)
	 */
	pptr = (__u16 *)&(((char *)ciph)[ciph->ihl*4]);

	/* Ensure the checksum is correct */
	if (ip_compute_csum((unsigned char *) icmph, len))
	{
		/* Failed checksum! */
		IP_VS_DBG(1, "forward ICMP: failed checksum from %d.%d.%d.%d!\n",
			  NIPQUAD(iph->saddr));
		return NF_DROP;
	}

	IP_VS_DBG(11, "Handling outgoing ICMP for "
                  "%u.%u.%u.%u:%d -> %u.%u.%u.%u:%d\n",
		  NIPQUAD(ciph->saddr), ntohs(pptr[0]),
		  NIPQUAD(ciph->daddr), ntohs(pptr[1]));

	/* ciph content is actually <protocol, caddr, cport, daddr, dport> */
	cp = ip_vs_conn_out_get(ciph->protocol, ciph->daddr, pptr[1],
				ciph->saddr, pptr[0]);
	if (cp == NULL)
		return NF_ACCEPT;

	if (IP_VS_FWD_METHOD(cp) != 0) {
		IP_VS_ERR("shouldn't reach here, because the box is on the"
                          "half connection in the tun/dr module.\n");
	}

	/* Now we do real damage to this packet...! */
	/* First change the source IP address, and recalc checksum */
	iph->saddr = cp->vaddr;
	ip_send_check(iph);

	/* Now change the *dest* address in the contained IP */
	ciph->daddr = cp->vaddr;
	ip_send_check(ciph);

	/* the TCP/UDP dest port - cannot redo check */
	pptr[1] = cp->vport;
	__ip_vs_conn_put(cp);

	/* And finally the ICMP checksum */
	icmph->checksum = 0;
	icmph->checksum = ip_compute_csum((unsigned char *) icmph, len);

	IP_VS_DBG(11, "Forwarding correct outgoing ICMP to "
                  "%u.%u.%u.%u:%d -> %u.%u.%u.%u:%d\n",
		  NIPQUAD(ciph->saddr), ntohs(pptr[0]),
		  NIPQUAD(ciph->daddr), ntohs(pptr[1]));

	skb->nfmark = NFMARK_IPVS_PROPERTY;

	return NF_ACCEPT;
}


/*
 *	It is hooked at the NF_IP_FORWARD chain, used only for VS/NAT.
 *	Check if outgoing packet belongs to the established ip_vs_conn,
 *      rewrite addresses of the packet and send it on its way...
 */
static unsigned int ip_vs_out(unsigned int hooknum,
                              struct sk_buff **skb_p,
                              const struct net_device *in,
                              const struct net_device *out,
                              int (*okfn)(struct sk_buff *))
{
	struct sk_buff  *skb = *skb_p;
	struct iphdr	*iph;
	union ip_vs_tphdr h;
	struct ip_vs_conn *cp;
	int size;
        int doff;                       /* transport protocol data offset */
        int csum;                       /* checksum */
        int csum_ok;                    /* csum_ok says if csum is valid */

	EnterFunction(11);

	if (skb->nfmark == NFMARK_IPVS_PROPERTY)
                return NF_ACCEPT;

	iph = skb->nh.iph;
	h.raw = (char*) iph + iph->ihl * 4;
	size = ntohs(iph->tot_len) - (iph->ihl * 4);
	doff = ip_vs_proto_doff(iph->protocol, h.raw, size);

	switch (doff) {
	case 0:
		/*
		 *	Input path: other IP protos Ok, will
		 *	reach local sockets path.
		 */
		return NF_ACCEPT;
	case -1:
		IP_VS_DBG(0, "I-pkt invalid packet data size\n");
		return NF_DROP;
	}

	if (iph->protocol == IPPROTO_ICMP)
		return ip_vs_out_icmp(skb_p);

        IP_VS_DBG(11, "O-pkt: %s size=%d\n",
                  ip_vs_proto_name(iph->protocol), size);
                
        /*
         *	Now hunt the list to see if we have an old entry
         */
        cp = ip_vs_conn_out_get(iph->protocol, iph->saddr, h.portp[0],
                                iph->daddr, h.portp[1]);
        if (!cp) {
		if (ip_vs_lookup_real_service(iph->protocol,
                                              iph->saddr, h.portp[0])) {
			/*
			 * Notify the real server: there is no existing
                         * entry if it is not RST packet or not TCP packet.
			 */
			if (!h.th->rst || iph->protocol != IPPROTO_TCP) {
				icmp_send(skb, ICMP_DEST_UNREACH,
                                          ICMP_PORT_UNREACH, 0);
				kfree_skb(skb);
				return NF_STOLEN;
			}
		}
                IP_VS_DBG(12, "packet for %s %d.%d.%d.%d:%d "
                          "continue traversal as normal.\n",
                          ip_vs_proto_name(iph->protocol),
                          NIPQUAD(iph->daddr),
                          ntohs(h.portp[1]));
                return NF_ACCEPT;
        }

        /* Check if the checksum is OK, when it is not UDP or check!=0 */
	csum = 0;
	csum_ok = 0;
        if ((iph->protocol != IPPROTO_UDP) || (h.uh->check != 0)) {
                switch (skb->ip_summed)
                {
                case CHECKSUM_NONE:
                        csum = csum_partial(h.raw + doff, size - doff, 0);
                        IP_VS_DBG(11, "O-pkt: %s I-datacsum=%d\n",
                                  ip_vs_proto_name(iph->protocol), csum);
                        skb->csum = csum_partial(h.raw , doff, csum);
                        csum_ok++;
                        break;

                case CHECKSUM_HW:
                        if (csum_tcpudp_magic(iph->saddr, iph->daddr, size,
                                              iph->protocol, skb->csum)) {
                                IP_VS_DBG(0, "Outgoing failed %s checksum "
                                          "from %d.%d.%d.%d (size=%d)!\n",
                                          ip_vs_proto_name(iph->protocol),
                                          NIPQUAD(iph->saddr),
                                          size);
                                return NF_DROP;
                        }
                default:
                        /* CHECKSUM_UNNECESSARY */
                }
        }

        IP_VS_DBG(11, "Outgoing %s %u.%u.%u.%u:%d->%u.%u.%u.%u:%d\n",
                  ip_vs_proto_name(iph->protocol),
                  NIPQUAD(iph->saddr), ntohs(h.portp[0]),
                  NIPQUAD(iph->daddr), ntohs(h.portp[1]));

 	/*
 	 *	Change the fragments origin
 	 */
 	size = skb->len - (h.raw - skb->nh.raw);

        /*
         *	Set iph addr and port from ip_vs obj.
         */
 	iph->saddr = cp->vaddr;
 	h.portp[0] = cp->vport;

	/*
	 *	Invalidate csum saving if tunnel has connection helper
	 */
	if (cp->app)
		csum_ok = 0;

 	/*
 	 *	Attempt ip_vs_app call.
         *	will fix ip_vs and iph seq stuff
 	 */
        if (ip_vs_app_pkt_out(cp, skb_p) != 0) {
                /*
                 *	skb has possibly changed, update pointers.
                 */
                skb = *skb_p;
                iph = skb->nh.iph;
		h.raw = (char*) iph + iph->ihl *4;
                size = skb->len - (h.raw - skb->nh.raw);
		/* doff should have not changed */
        }

 	/*
 	 *	Adjust packet accordingly to protocol
 	 */
	if (!csum_ok) {
                /* transport's payload partial csum */
		csum = csum_partial(h.raw + doff, size - doff, 0);
	}
	skb->csum = csum;

	IP_VS_DBG(11, "O-pkt: %s size=%d O-datacsum=%d\n",
                  ip_vs_proto_name(iph->protocol), size, csum);

	/*
	 * 	Protocol csum
	 */
	switch (iph->protocol) {
        case IPPROTO_TCP:
                h.th->check = 0;
                h.th->check=csum_tcpudp_magic(iph->saddr, iph->daddr,
                                              size, iph->protocol,
                                              csum_partial(h.raw, doff, csum));
                IP_VS_DBG(11, "O-pkt: %s O-csum=%d (+%d)\n",
                          ip_vs_proto_name(iph->protocol), h.th->check,
                          (char*) & (h.th->check) - (char*) h.raw);
                break;
        case IPPROTO_UDP:
                h.uh->check = 0;
                h.uh->check=csum_tcpudp_magic(iph->saddr, iph->daddr,
                                              size, iph->protocol,
                                              csum_partial(h.raw, doff, csum));
                if (h.uh->check == 0)
                        h.uh->check = 0xFFFF;
                IP_VS_DBG(11, "O-pkt: %s O-csum=%d (+%d)\n",
                          ip_vs_proto_name(iph->protocol), h.uh->check,
                          (char*) &(h.uh->check)- (char*) h.raw);
                break;
	}
	ip_send_check(iph);

	ip_vs_set_state(cp, VS_STATE_OUTPUT, iph, h.portp);
	ip_vs_conn_put(cp);
        
	skb->nfmark = NFMARK_IPVS_PROPERTY;

        LeaveFunction(11);
	return NF_ACCEPT;
}


unsigned int check_for_ip_vs_out(struct sk_buff **skb_p,
                                 int (*okfn)(struct sk_buff *))
{
        unsigned int ret;
        
        ret = ip_vs_out(NF_IP_FORWARD, skb_p, NULL, NULL, NULL);
        if (ret != NF_ACCEPT) {
                return ret;
        } else {
                if ((*skb_p)->nfmark == NFMARK_IPVS_PROPERTY) {
                        (*okfn)(*skb_p);
                        return NF_STOLEN;
                }
        }
        return NF_ACCEPT;
}

        
/*
 *	Handle ICMP messages in the outside-to-inside direction (incoming).
 *	Find any that might be relevant, check against existing connections,
 *	forward to the right destination host if relevant.
 *	Currently handles error types - unreachable, quench, ttl exceeded
 */
static int ip_vs_in_icmp(struct sk_buff **skb_p)
{
        struct sk_buff 	*skb   = *skb_p;
 	struct iphdr	*iph   = skb->nh.iph;
	struct icmphdr  *icmph = (struct icmphdr *)((char *)iph+(iph->ihl<<2));
	struct iphdr    *ciph;	/* The ip header contained within the ICMP */
	__u16	        *pptr;	/* port numbers from TCP/UDP contained header */
	unsigned short   len   = ntohs(iph->tot_len) - (iph->ihl * 4);
	struct ip_vs_conn *cp;
        struct rtable *rt;			/* Route to the other host */
	struct ip_vs_dest *dest;
        u8     tos = iph->tos;
	u32    rtos;
        int    mtu;

 	IP_VS_DBG(12, "icmp in (%d,%d) %u.%u.%u.%u -> %u.%u.%u.%u\n",
                  icmph->type, ntohs(icmp_id(icmph)),
                  NIPQUAD(iph->saddr), NIPQUAD(iph->daddr));

        if ((icmph->type != ICMP_DEST_UNREACH) &&
            (icmph->type != ICMP_SOURCE_QUENCH) &&
            (icmph->type != ICMP_TIME_EXCEEDED))
                return NF_ACCEPT;

        /*
	 * If we get here we have an ICMP error of one of the above 3 types
	 * Now find the contained IP header
	 */
	ciph = (struct iphdr *) (icmph + 1);

	/* We are only interested ICMPs generated from TCP or UDP packets */
	if ((ciph->protocol != IPPROTO_UDP) &&
	    (ciph->protocol != IPPROTO_TCP))
		return NF_ACCEPT;

	/* Ensure the checksum is correct */
	if (ip_compute_csum((unsigned char *) icmph, len))
	{
		/* Failed checksum! */
		IP_VS_ERR("incoming ICMP: failed checksum from %d.%d.%d.%d!\n",
                          NIPQUAD(iph->saddr));
		return NF_DROP;
	}

        pptr = (__u16 *)&(((char *)ciph)[ciph->ihl*4]);

 	IP_VS_DBG(11, "Handling incoming ICMP for "
                  "%u.%u.%u.%u:%d -> %u.%u.%u.%u:%d\n",
                  NIPQUAD(ciph->saddr), ntohs(pptr[0]),
                  NIPQUAD(ciph->daddr), ntohs(pptr[1]));

	/* This is pretty much what ip_vs_conn_in_get() does,
           except parameters are wrong way round */
        cp = ip_vs_conn_in_get(ciph->protocol,
                               ciph->daddr, pptr[1],
                               ciph->saddr, pptr[0]);
	if (cp == NULL)
		return NF_ACCEPT;

        /* The ICMP packet for VS/TUN, VS/DR and LOCALNODE will be
           forwarded directly here, because there is no need to
           translate address/port back */
        if (IP_VS_FWD_METHOD(cp) != IP_VS_CONN_F_MASQ) {
                int ret;
                if (cp->packet_xmit)
                        ret = cp->packet_xmit(skb, cp);
                else
                        ret = NF_ACCEPT;
                ip_vs_conn_put(cp);
                return ret;
        }

	/* The ICMP packet for VS/NAT must be written to correct addresses
           before being forwarded to the right server */
        if ((skb=vs_skb_cow(skb_p, &iph, (unsigned char**)&icmph)) == NULL) {
		__ip_vs_conn_put(cp);
		return NF_DROP;
	}

	ciph = (struct iphdr *) (icmph + 1);
	pptr = (__u16 *)&(((char *)ciph)[ciph->ihl*4]);

	/* Now we do real damage to this packet...! */
	/* First change the dest IP address, and recalc checksum */
	iph->daddr = cp->daddr;
	ip_send_check(iph);

	/* Now change the *source* address in the contained IP */
	ciph->saddr = cp->daddr;
	ip_send_check(ciph);

	/* the TCP/UDP source port - cannot redo check */
	pptr[0] = cp->dport;
	__ip_vs_conn_put(cp);

	/* And finally the ICMP checksum */
	icmph->checksum = 0;
	icmph->checksum = ip_compute_csum((unsigned char *) icmph, len);

 	IP_VS_DBG(11, "Forwarding incoming ICMP to "
                  "%u.%u.%u.%u:%d -> %u.%u.%u.%u:%d\n",
                  NIPQUAD(ciph->saddr), ntohs(pptr[0]),
                  NIPQUAD(ciph->daddr), ntohs(pptr[1]));

        /*
         * Send the mangled packet immediately here
         */
	dest = cp->dest;
	rtos = RT_TOS(tos);
	spin_lock(&dest->dst_lock);
	if (!sysctl_use_dst || !(rt = (struct rtable *) __ip_vs_dst_check(dest, rtos, 0))) {
		if (ip_route_output(&rt, dest->addr, 0, rtos, 0)) {
			spin_unlock(&dest->dst_lock);
			IP_VS_DBG(0, "ip_vs_in_imcp(): route error, "
				  "dest: %u.%u.%u.%u\n", NIPQUAD(dest->addr));
			goto tx_error_icmp;
		}
		__ip_vs_dst_set(dest, rtos, dst_clone(&rt->u.dst));
		IP_VS_DBG(10, "ICMP: new dst %u.%u.%u.%u, refcnt=%d, rtos=%X\n",
                          NIPQUAD(dest->addr), atomic_read(&rt->u.dst.__refcnt),
                          rtos);
        }
	spin_unlock(&dest->dst_lock);

        /* MTU checking, is it necessary here??? */
        mtu = rt->u.dst.pmtu;
        if ((skb->len > mtu) && (iph->frag_off&__constant_htons(IP_DF))) {
                icmp_send(skb, ICMP_DEST_UNREACH,ICMP_FRAG_NEEDED, htonl(mtu));
                ip_rt_put(rt);
                IP_VS_DBG(0, "ip_vs_in_icmp(): frag needed\n");
                goto tx_error;
        }

        /* drop old route */
        dst_release(skb->dst);
        skb->dst = &rt->u.dst;

#ifdef CONFIG_NETFILTER_DEBUG
        skb->nf_debug = 1 << NF_IP_LOCAL_OUT;
#endif /* CONFIG_NETFILTER_DEBUG */
        ip_send(skb);
        return NF_STOLEN;

  tx_error_icmp:
	dst_link_failure(skb);
  tx_error:
	dev_kfree_skb(skb);
	return NF_STOLEN;
}


/*
 *	Check if it's for virtual services, look it up,
 *	and send it on its way...
 */
static unsigned int ip_vs_in(unsigned int hooknum,
			     struct sk_buff **skb_p,
			     const struct net_device *in,
			     const struct net_device *out,
			     int (*okfn)(struct sk_buff *))
{
	struct sk_buff	*skb = *skb_p;
	struct iphdr	*iph = skb->nh.iph;
	union ip_vs_tphdr h;
	struct ip_vs_conn *cp;
	unsigned short size;
	int doff;
	struct ip_vs_service *svc;

	/*
	 *	Big tappo: only PACKET_HOST (nor loopback neither mcasts)
	 *	... don't know why 1st test DOES NOT include 2nd (?)
	 */
	if (skb->pkt_type != PACKET_HOST || skb->dev == &loopback_dev) {
		IP_VS_DBG(12, "packet type=%d proto=%d daddr=%d.%d.%d.%d ignored\n",
			  skb->pkt_type,
			  iph->protocol,
			  NIPQUAD(iph->daddr));
		return NF_ACCEPT;
	}

	h.raw = (char*) iph + iph->ihl * 4;

	/*
	 *	IP payload size
	 */
	size = ntohs(iph->tot_len) - (iph->ihl * 4);
	doff = ip_vs_proto_doff(iph->protocol, h.raw, size);

	switch (doff) {
	case 0:
		/*
		 *	Input path: other IP protos Ok, will
		 *	reach local sockets path.
		 */
		return NF_ACCEPT;
	case -1:
		IP_VS_DBG(0, "I-pkt invalid packet data size\n");
		return NF_DROP;
	}

	if (iph->protocol == IPPROTO_ICMP)
		return(ip_vs_in_icmp(skb_p));

	IP_VS_DBG(11, "Incoming %s %u.%u.%u.%u:%d->%u.%u.%u.%u:%d\n",
		  ip_vs_proto_name(iph->protocol),
		  NIPQUAD(iph->saddr), ntohs(h.portp[0]),
		  NIPQUAD(iph->daddr), ntohs(h.portp[1]));

	/*
	 * reroute to original host:port if found...
	 */
	cp = ip_vs_conn_in_get(iph->protocol, iph->saddr, h.portp[0],
			       iph->daddr, h.portp[1]);

	/*
	 * Checking the server status
	 */
	if (cp && cp->dest && !(cp->dest->flags & IP_VS_DEST_F_AVAILABLE)) {
		/*
		 * If the dest is not avaiable, don't restart the timer
		 * of the packet, but silently drop it.
		 */
		add_sltimer(&cp->timer);
		__ip_vs_conn_put(cp);
		return NF_DROP;
	}

	if (!cp
            && (h.th->syn || (iph->protocol!=IPPROTO_TCP))
            && (svc = ip_vs_service_get(skb->nfmark, iph->protocol,
                                        iph->daddr, h.portp[1]))) {
                
		if (ip_vs_todrop()) {
			/*
			 * It seems that we are very loaded.
			 * We have to drop this packet :(
			 */
                        ip_vs_service_put(svc);
			return NF_DROP;
		}
                
		/*
		 * Let the virtual server select a real server for the
                 * incomming connection, and create a connection entry.
		 */
		cp = ip_vs_schedule(svc, iph);
		if (!cp)
			return ip_vs_leave(svc, skb);
                ip_vs_service_put(svc);
	}

	if (cp != NULL) {
                int ret;

                ip_vs_set_state(cp, VS_STATE_INPUT, iph, h.portp); 
                if (cp->packet_xmit)
                        ret = cp->packet_xmit(skb, cp);
                else {
                        IP_VS_DBG(0, "warning: packet_xmit is null");
                        ret = NF_ACCEPT;
                }
                ip_vs_conn_put(cp);
                return ret;
	}

	/* sorry, all this trouble for a no-hit :) */
        IP_VS_DBG(12, "packet for %s %d.%d.%d.%d:%d continue traversal "
                  "as normal.\n",
                  ip_vs_proto_name(iph->protocol),
                  NIPQUAD(iph->daddr),
                  ntohs(h.portp[1]));

	return NF_ACCEPT;
}


/* After packet filtering, forward packet through VS/DR, VS/TUN,
   or VS/NAT(change destination), so that filtering rules can be
   applied to IPVS. */
static struct nf_hook_ops ip_vs_in_ops = {
        { NULL, NULL },
        ip_vs_in, PF_INET, NF_IP_LOCAL_IN, 100
};

/* After packet filtering, change source only for VS/NAT */
static struct nf_hook_ops ip_vs_out_ops = {
        { NULL, NULL },
        ip_vs_out, PF_INET, NF_IP_FORWARD, 100
};

/* Before the netfilter connection tracking, exit from POST_ROUTING */
static struct nf_hook_ops ip_vs_post_routing_ops = {
        { NULL, NULL },
        ip_vs_post_routing, PF_INET, NF_IP_POST_ROUTING, NF_IP_PRI_NAT_SRC-1
};


/*
 *	Initialize IP Virtual Server
 */
static int __init ip_vs_init(void)
{
	int ret;

        ret = ip_vs_control_init();
	if (ret < 0) {
		IP_VS_ERR("can't setup control.\n");
		goto cleanup_nothing;
	}

	ip_vs_sltimer_init();

	ret = ip_vs_conn_init();
	if (ret < 0) {
		IP_VS_ERR("can't setup connection table.\n");
		goto cleanup_sltimer;
	}

	ret = ip_vs_app_init();
	if (ret < 0) {
		IP_VS_ERR("can't setup application helper.\n");
		goto cleanup_conn;
	}

	ret = nf_register_hook(&ip_vs_in_ops);
	if (ret < 0) {
		IP_VS_ERR("can't register in hook.\n");
		goto cleanup_app;
	}
	ret = nf_register_hook(&ip_vs_out_ops);
	if (ret < 0) {
		IP_VS_ERR("can't register out hook.\n");
		goto cleanup_inops;
	}
	ret = nf_register_hook(&ip_vs_post_routing_ops);
	if (ret < 0) {
		IP_VS_ERR("can't register post_routing hook.\n");
		goto cleanup_outops;
	}

        IP_VS_INFO("ipvs loaded.\n");
	return ret;

  cleanup_outops:
	nf_unregister_hook(&ip_vs_out_ops);
  cleanup_inops:
	nf_unregister_hook(&ip_vs_in_ops);
  cleanup_app:
	ip_vs_app_cleanup();
  cleanup_conn:
	ip_vs_conn_cleanup();
  cleanup_sltimer:
	ip_vs_sltimer_cleanup();
	ip_vs_control_cleanup();
  cleanup_nothing:
	return ret;
}

static void __exit ip_vs_cleanup(void)
{
	nf_unregister_hook(&ip_vs_post_routing_ops);
	nf_unregister_hook(&ip_vs_out_ops);
	nf_unregister_hook(&ip_vs_in_ops);
	ip_vs_app_cleanup();
	ip_vs_conn_cleanup();
	ip_vs_sltimer_cleanup();
	ip_vs_control_cleanup();
        IP_VS_INFO("ipvs unloaded.\n");
}

module_init(ip_vs_init);
module_exit(ip_vs_cleanup);
