/*
 * IPVS:        Weighted Least-Connection Scheduling module
 *
 * Version:     $Id: ip_vs_wlc.c,v 1.3 2000/07/06 14:57:09 wensong Exp $
 *
 * Authors:     Wensong Zhang <wensong@iinchina.net>
 *              Peter Kese <peter.kese@ijs.si>
 *
 *              This program is free software; you can redistribute it and/or
 *              modify it under the terms of the GNU General Public License
 *              as published by the Free Software Foundation; either version
 *              2 of the License, or (at your option) any later version.
 *
 * Changes:
 *     Wensong Zhang            :     changed the ip_vs_wlc_schedule to return dest
 *     Wensong Zhang            :     changed to use the inactconns in scheduling
 *     Wensong Zhang            :     changed some comestics things for debugging
 *     Wensong Zhang            :     changed for the d-linked destination list
 *     Wensong Zhang            :     added the ip_vs_wlc_update_svc
 *     Wensong Zhang            :     added any dest with weight=0 is quiesced 
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/errno.h>

#include "ip_vs.h"
//#include <net/ip_vs.h>


static int
ip_vs_wlc_init_svc(struct ip_vs_service *svc)
{
	return 0;
}


static int
ip_vs_wlc_done_svc(struct ip_vs_service *svc)
{
	return 0;
}


static int
ip_vs_wlc_update_svc(struct ip_vs_service *svc)
{
	return 0;
}


/*
 *    Weighted Least Connection scheduling
 */
static struct ip_vs_dest *
ip_vs_wlc_schedule(struct ip_vs_service *svc)
{
	register struct list_head *l, *e;
	struct ip_vs_dest *dest, *least;
	int loh, doh;

	IP_VS_DBG(1, "ip_vs_wlc_schedule(): Scheduling...\n");

	l = &svc->destinations;
	if (l == l->next)
		return NULL;

	/*
	 * We think the overhead of processing active connections is fifty
	 * times than that of inactive conncetions in average. (This fifty
	 * times might be not accurate, we will change it later.) We use
	 * the following formula to estimate the overhead:
	 *		  dest->activeconns*50 + dest->inactconns
	 * and the load:
	 *		  (dest overhead) / dest->weight
	 *
	 * Remember -- no floats in kernel mode!!!
	 * The comparison of h1*w2 > h2*w1 is equivalent to that of
	 *		  h1/w1 > h2/w2
	 * if every weight is larger than zero.
	 *
	 * The server with weight=0 is quiesced and will not receive any
	 * new connection.
	 */

	for (e=l->next; e!=l; e=e->next) {
		least = list_entry(e, struct ip_vs_dest, n_list);
		if (atomic_read(&least->weight) > 0) {
			loh = atomic_read(&least->activeconns) * 50
				+ atomic_read(&least->inactconns);
			goto nextstage;
		}
	}
	return NULL;
	
	/*
	 *    Find the destination with the least load.
	 */
  nextstage:
	for (e=e->next; e!=l; e=e->next)
	{
		dest = list_entry(e, struct ip_vs_dest, n_list);
		doh = atomic_read(&dest->activeconns) * 50
			+ atomic_read(&dest->inactconns);
		if (loh * atomic_read(&dest->weight) >
                    doh * atomic_read(&least->weight))
		{
			least = dest;
			loh = doh;
		}
	}

	IP_VS_DBG(1, "WLC: server %u.%u.%u.%u:%u "
                  "activeconns %d refcnt %d weight %d overhead %d\n",
                  NIPQUAD(least->addr), ntohs(least->port),
                  atomic_read(&least->activeconns),
                  atomic_read(&least->refcnt),
                  atomic_read(&least->weight), loh);

	return least;
}


static struct ip_vs_scheduler ip_vs_wlc_scheduler =
{
	{0},			/* n_list */
	"wlc",			/* name */
	ATOMIC_INIT(0),         /* refcnt */
	THIS_MODULE,		/* this module */
	ip_vs_wlc_init_svc,	/* service initializer */
	ip_vs_wlc_done_svc,	/* service done */
	ip_vs_wlc_update_svc,	/* service updater */
	ip_vs_wlc_schedule,	/* select a server from the destination list */
};


#ifdef MODULE
int ip_vs_wlc_init(void)
#else
int __init ip_vs_wlc_init(void)
#endif
{
	IP_VS_INFO("Initializing WLC scheduling\n");
	INIT_LIST_HEAD(&ip_vs_wlc_scheduler.n_list);
	return register_ip_vs_scheduler(&ip_vs_wlc_scheduler);
}


#ifdef MODULE
EXPORT_NO_SYMBOLS;

int
init_module(void)
{
	return ip_vs_register_scheduler_module(&ip_vs_wlc_scheduler);
}

void
cleanup_module(void)
{
	ip_vs_unregister_scheduler_module(&ip_vs_wlc_scheduler);
}

#endif /* MODULE */
