#!/usr/local/bin/perl
#configure-2.2.15-lvs.pl v0.6 (C) Joseph Mack 1999-2000 jmack@wm7d.net
#
#released under GPL
#This code is part of the Linux Virtual Server (LVS) project
#http://www.linuxvirtualserver.org
#
#v0.1 6 Nov 99 for kernel 2.2.12
#v0.2 7 Nov 99 for kernel 2.2.13
#v0.3 14 Nov 99 added option for varying timeout for ppc connnection
#v0.4 17 Nov 99 fixed bug in ppc setup code
#v0.5 Jan 00 1.Changed some names to make more readable ( eg SERVER_LVS_DEVICE to SERVER_VIP_DEVICE)
#            2.rc.lvs file now has VIP (and a few other variables) as $VIP rather than values
#            (makes it easier to edit the rc.lvs file)
#            3.with VS-NAT old default gw is deleted before new one added.
#v0.6 Jun 00 reads all input before doing anything. 
#            This allows me to do things dependant on two simultaneous conditions 
#            (and makes the order of input lines irrelevant)
#            The code no longer can report errors by input line, as the code has read to the end
#            of the input file before it starts to parse.
#            added rc.horms, rc.ratz
#
#Caveat - not all inputs have been tested. 
#Please send bug reports to Joe jmack@wm7d.net
#
#----------------------------------------------------
#Description:
#
#	Generates rc.d type files for an LVS in which the 
#	director is running a Linux 2.2.x kernel.
#	(the previous version of this file, configure.lvs was for a 
#	director running a Linux 2.0.36 kernel, and was a shell script)
#
#	Input - a conf file which has
#		1. The type of LVS (VS_NAT, VS_TUN or VS_DR)
#		2. The IPs for the director and the servers
#		3. The services being served 
#
#	Outputs
#		1. rc.lvs_xxx file, which you copy (or nfs mount) to /etc/rc.d on all machines 
#		2. mon_xxx.cf file, which (if you are using mon) you copy to /etc/mon on the director 
#			where xxx is nat, tun or dr
#
#	For information about monitors see the LVS-HOWTO 
#		at http://www.linuxvirtualserver.org
#	and the 
#		mon site http://www.kernel.org/software/mon/ 
#	The fping monitor will be chosen if there is no monitor for your service.
#	Available monitors:
#		dialin, dns, fping, freespace, ftp, ftp_kernelorg, hpnp, http, http_t, 
#		imap, ldap, msql-mysql, na_quota, netappfree,
#		nntp, ping, pop3, process, rd, reboot, ssh, seq, smtp, tcp, telnet, up_rtt
#	Please read the "Big Caveat" about monitors in the LVS-HOWTO 
#
#----------------------------------------------------------

=head1 NAME

B<configure_lvs.pl> - writes an rc.lvs and mon.cf file from information in a conf file.

=head1 LVS PROJECT

The Linux Virtual Server (LVS) is a project headed by 
Wensong Zhang, which is writing a highly available layer 4 switch. 
All information (code, documentation, HOWTO, mailing lists) is
at http://www.linuxvirtualserver.org

=head1 SYNOPSYS

B<configure_lvs.pl> I<conf-file>

where I<conf-file> (e.g. lvs_dr.conf) 
is a file containing the IP/hostnames of the machines in 
the LVS and the services that are switched.  
This results in two files, rc.lvs and mon.cf -

=head2 rc.lvs

B<rc.lvs> is an executable which configures the LVS. 
It is first run on the on the director 
(B<./rc.lvs>)
and then the realservers (it knows whether it's running
on a director or a realserver).

B<rc.lvs> is idempotent. ie you can run it them multiple times
and the machine will have the correct state after each run 
(it clears old settings before adding new ones).

The director is a Linux box (kernel 2.4 and
2.2 supported, 2.0 is no longer supported).

The realservers can have almost any OS (see HOWTO). 
Realservers running non-Linux unices are configured by code 
contributed by Ratz <ratz@tac.ch>. 
This code has not been well tested.
Any updates, queries etc, please contact Ratz or myself.

=head2 mon.cf 

is placed in /etc/mon/ on the director and is read by mon at mon startup.

B<mon.cf> configures monitors and alerts to check the state of services being 
offered by the LVS realservers, and is used to add/remove services to the LVS.
This file is not needed to setup a working LVS. It is only used if you
are using mon to monitor the state of the LVS for failover (see the HOWTO).
If you're unsure what it's for, you can ignore it.

(Note: I haven't changed or tested the mon.cf file for this revision.
It seems that no-one is using it. It may be broken. Let me know if
anyone is actually using it).

=head1 INSTALL

B<configure_lvs.pl> is run on the director
(where it detects the director's kernel version)
and can be run in any directory.
The resulting two rc files are copied/exported (I export them) 
to the realservers:/directory in the
LVS where they are run to configure the LVS.

=head1 SYSTEM REQUIREMENTS

The configure script has been tested on Linux/perl-5.00503 and 5.6.0.

=head1 SOFTWARE REQUIREMENTS:

You will need B<fping> installed on all the nodes in the LVS.
B<fping> is part of the mon package. 

Get B<fping> from

	ftp://ftp.kernel.org/pub/software/admin/mon
	http://www.kernel.org/software/mon/


=head1 FILES:

You get the following files 

B<configure_lvs.pl>

lvs.conf - sample conf files to feed to configure_lvs.pl

rc.lvs - sample output files produced from the conf files.

=head1 DESCRIPTION

B<configure_lvs> generates rc.lvs which 

=head2 sets LVS type 

VS-NAT, VS-DR, VS-TUN

=head2 sets IPs and network on director 

including VIP. Director can accept packets on VIP
by a network device (eg eth0) or by transparent proxy. sets
default gw.

=head2 sets up services 

using ipvsadm on director 

=head2 sets IPs and network on realserver(s) 

including non-arp'ing VIP. Realservers can accept packets on VIP
by a network device (eg eth0 or lo:0) device or by transparent proxy. 
sets default gw

=head2 tests setup

as far as is possible from inside the LVS. Many common
problems can be trapped here.

B<configure_lvs> generates rc.mon which 

=head2 sets up mon

The LVS can be started in the off state and as the realservers
bootup/come on line, the LVS will detect their arrival and
put them into the LVS. Failure of a service on a realserver
will be detected and this service will be removed from the LVS
list of offered services, till the service comes back on line.

=head1 WHICH IS THE DIRECTOR?

answer: For the moment the director is the one with the 
director_inside_ip (DIIP).

reason: Although the network connectivity is different 
for the director and the realservers in an LVS, it is 
(conceptually at least) possible to promote a realserver 
to a director following a director failure. All the 
machines in an LVS then should be able to be configured 
as peers and have one machine promoted to director.

In this case, the rc.lvs has to be able to tell 
whether it's running on a director. The Linux-HA setup of LVS 
(eg ultramonkey at http://ultramonkey.sourceforge.net) 
puts flags into files in /etc/rc.d which 
results in a floating or heatbeat IP being 
assigned to the director. (This IP becomes 
the default gw for the realservers when 
running under VS-NAT.) 

B<configure_lvs> will read these flags 
eventually but for the moment B<configure_lvs> 
determines whether it is running on a director
by looking for the I<director_inside_IP> (DIIP). 
This DIIP is set in lvs.conf and is on the 
LVS network (ie the network the realservers connect to).
(Depending on your setup, this IP may or 
may not be accessable from the internet.)

Before you run B<configure_lvs>, you must 
add this IP to the machine you want to be 
the director (on say an aliased ethernet device).

=head1 EXAMPLE FILES FOR THE IMPATIENT

1. 
The file B<lvs_dr.conf.simple_telnet> will setup a one realserver 
LVS. You need a client with telnet (any OS), a director (Linux) and
a realserver (any OS but for a test Linux will be faster to setup)
all on the same network. Edit B<lvs_dr.conf.simple_telnet> changing
the names of the machines to your setup. Make the client the default gw
of the director and the realservers.

Alternately, change telnet to http to connect from the browser on the
client to an httpd on the realserver.

2. 
The file B<lvs_nat.conf.simple_telnet> will setup a one realserver 
LVS as above using VS-NAT. The network for the realserver must be private
(eg 192.168.1.0/24) and the realserver must not be able to route to the
client.  

=head1 /etc/hosts

The I<conf-file> accepts hostnames (hostname or hostname.domain.org
format) in any place in I<conf-file> that an IP is expected. If IPs
are resolvable (eg you have an /etc/hosts file on the LVS machines),
then hostnames will be output to the rc.lvs file.
 

here is my /etc/hosts. It contains the normal names for the hosts
and the aliases used for the lvs (director*, realserver*).

	127.0.0.1	localhost

	192.168.1.9	director-inside.mack.net 	director-inside happy
	192.168.1.11	sneezy.mack.net 	sneezy		realserver1 
	192.168.1.12	bashfull.mack.net	bashfull	realserver2
	192.168.1.13	dopey.mack.net		dopey		realserver3
	192.168.1.14	doc.mack.net		doc		realserver4
	192.168.1.15	sleepy.mack.net		sleepy		realserver5
	192.168.1.110	lvs.mack.net		lvs
	192.168.2.110	lvs2.mack.net		lvs2
	
	192.168.2.1	director.mack.net	director-outside
	192.168.2.254	client2.mack.net	client2
	192.168.1.254	client.mack.net		client

=head1 /etc/services

The I<conf-file> script will accept the service name (eg telnet)
anywhere a port number (eg 23) is expected
using the /etc/services file to translate one to another. 
If the /etc/services file has a port->name translation, then
the service name will be output to the I<conf-file>.

Likely additions needed to your /etc/services will be

	ftp-data        20/tcp
	ssh             22/tcp
	domain          53/tcp  nameserver      dns #the string "dns" needs to be added here
	domain          53/tcp  nameserver      dns #and here
	shell           514/tcp cmd             rsh #add rsh here
	https           443/tcp
	https           443/udp
	printer         515/tcp spooler         lpd #add the string "lpd" here
	nfs             2049/udp        nfs
	mysql           3306/tcp
	mysql           3306/udp
	netpipe         5002/tcp

=head1 CLASSLESS NETWORKS

All networks/netmasks are set explicitely in the I<conf-file>, 
so presumably you can set networks on any boundary. 
I have only tested this with a class C setup. 

=head1 DESCRIPTION OF CONF FILE

The I<conf-file> consists of 
variable names with arguements, 
comments (any part of a line starting at '#'),
and
blank lines. 
Hostnames can be a resolvable name or a dotted quad IP. 
The variables can be in any order,
but for my sanity I keep them in the following sections and order

	lvs type
	director setup
	services setup
	realserver setup

Devices for all IPs are explicitely set. This allows the
script to work with directors and realservers which have 1,2..n NICs.

Here are the variables required, with examples.

	#LVS_TYPE: VS_TUN|VS_NAT|VS_DR 
			#(string is matched with tun/nat/dr, 
			# any string containing tun, nat or dr will do. eg tunnel, natural, drive)
LVS_TYPE=VS_DR


	#INITIAL_STATE: on|off  
	#Only useful if you are using mon, otherwise ignore 
	#(leave "on" or ommit, or comment out)
	#
	#suggested: for testing - "on", for production systems - "off", 
	#if you don't understand the following then choose "on"
	#
	#when testing LVS, the services (eg http, telnet) are already running 
	#on the real servers, before you start adding servers:services 
	#with ipvsadm on the director, and before you make a request from the client, 
	#
	#In a production system, if ipvsadm adds servers:services to
	#the director as soon as the director boots, the servers 
	#could still be booting/fsck'ing. 
	#A client could be routed to a server which is still coming up.
	#If you are running mon (and mon has been tested by pulling 
	#and reinserting cables etc), you can handle this by configuring
	#LVS with no entries (or entries with weight = 0).
	#Then when the director comes on line, the ipvsadm table will be empty.
	#Mon will also be running and as each server comes on line, mon will
	#detect an on-line server and add entries to the LVS
	#table by running ipvsadm, bringing the server into the LVS.
	#
	#since I don't have a big system to test this on, can someone let
	#me know if this actually works for big systems. - thanks Joe
	#
INITIAL_STATE=on  

	#director setup 
	#

	#Virtual IP (VIP)
	#
	#The LVS has its own IP (and name in DNS, presumably that of 
	#the cluster of servers, eg lvs.foobar.org).
	#This IP is called the VIP and is on a regular 
	#(or ip_aliased) ethernet device on the director.
	#Packets sent to the VIP are forwarded on to the 
	#servers according to the ipvsadm table.
	#The director will need a separate IP/machine name for 
	#logging in/admin, possibly on a different network, to the VIP. 
	#(eg if the LVS is forwards telnet packets, you will not be able
	#to telnet to the director). 
	#Clients must be able to route to/ping/connect with the VIP.
	#
	#For testing we use VIP=x.x.x.110 (where x.x.x.0 is your domain).
	#
	#After setup, clients must be about to route/ping/connect with this VIP.
	#
	#syntax: (Note the host mask and broadcast address).
	#The LVS can accept packets by a regular network device (eg eth0)
	#or by transparent proxy (see HOWTO)
	#I use an ethernet alias with the same number as the last number in the IP
	#(here 110) to reduce the chance of name space collisions
	#
	#VirtualIP: device, IP (or name), netmask, broadcast 
	#VIP=eth1:110 192.168.2.110 255.255.255.255 192.168.2.110
	#VIP=eth0:110 192.168.1.110 255.255.255.255 192.168.1.110
	#VIP=eth0:110 lvs 255.255.255.255 192.168.1.110
	#the director can accept packets by TP (tested for VS-DR so far)
	#VIP=TP 192.168.1.110 255.255.255.255 192.168.1.110
VIP=eth1:110 192.168.1.110 255.255.255.255 192.168.1.110

	#DIRECTOR_INSIDEIP
	#DIRECTOR_INSIDEIP: device IP(or name) network netmask broadcast
	#DIRECTOR_INSIDEIP=eth0:9 192.168.1.9 192.168.1.0 255.255.255.0 192.168.1.255
	#DIRECTOR_INSIDEIP=eth0:9 director_inside 192.168.1.0 255.255.255.0 192.168.1.255
DIRECTOR_INSIDEIP=eth0:9 192.168.1.9 192.168.1.0 255.255.255.0 192.168.1.255

	# DIRECTOR_DEFAULT_GW: hostname|ip
DIRECTOR_DEFAULT_GW=192.168.2.254

	#SERVICE:
	#syntax:
	#service proto (t|u) port service scheduling_algorithm (wlc|rr|wrr|pcc) server_IP[,weight (default 1)] [server IP[,weight]]
	#
	#For TUN, the realserver network can be anything, including the client's network.
	#For DR, the VIP must be part of the realserver network.
	#
	#the realserver IP's are on eth devices and must be accessable (ping etc) from the director
	#
	#scheduling algorithms 
	#wlc: weighted least connection
	#rr: round robin
	#wrr: weighted round robin
	#pcc: persistent client connection
	#
	#
	#telnet is LVS'ed using round robin scheduling to realserver x.x.x.10..15 and by localnode to the director
	#SERVICE=t telnet rr 192.168.1.10 192.168.1.11 192.168.1.12 192.168.1.13 192.168.1.14 192.168.1.15 127.0.0.1 
	#
	#dns requires 2 entries, one for udp and another for tcp
	#realserver x.x.x.12 gets twice the number of requests of realserver x.x.x.11
	#SERVICE=t dns rr 192.168.1.11,1 192.168.1.12,2  
	#SERVICE=u dns rr 192.168.1.11,1 192.168.1.12,2  
	#
SERVICE=t telnet rr sneezy 

SERVICE=t telnet rr bashfull 

SERVICE=t netpipe rr 192.168.1.11 

SERVICE=u nfs rr 192.168.1.11 

SERVICE=t rsh rr 192.168.1.11 


	#realserver setup 
	#
	#SERVER_VIP_DEVICE: TP|lo:x|tunl0|dummy0|eth0:1|...
	#TP on realservers is for VS-DR and VS-NAT only
	#lo:0 for VS-DR, tunl0 for VS-TUN
	#there is no VIP on the realservers for VS-NAT.
	#This input will be ignored for VS-NAT 
	#(you could also comment it out, or leave it out for VS-NAT) 
	#SERVER_VIP_DEVICE=TP
	#SERVER_VIP_DEVICE=tunl0
	#SERVER_VIP_DEVICE=dummy0
	#SERVER_VIP_DEVICE=eth0:1
SERVER_VIP_DEVICE=lo:0

	#SERVER_NET_DEVICE: eth0, eth1, hme0, le0:1
	#network device on the real servers (the device that has the IP's in SERVICE)
	#this device will have been setup by rc.inet1 before configuring LVS 
	#since all servers for convenience are configured identically, 
	#this will be the same for all servers
	#on linux eth0, eth1, eth0:1..., on Solaris le0, le0:1, hme0
	#SERVER_NET_DEVICE=hme0
	#SERVER_NET_DEVICE=le0:1
SERVER_NET_DEVICE=eth0

	#SERVER_DEFAULT_GW: name|ip
	#for vs-nat, must be on director
	#for vs_nat, must NOT be on director
	#for vs_tun, must NOT be on director,
	#but can be different for each realserver
	#null value will not install a default gw
	#and you can use the rc.d scripts on the host
	#to set the default gw
SERVER_DEFAULT_GW=192.168.1.254 	

#----------end lvs_dr.conf-------------

=head1 EXAMPLE I<conf-files>

lvs_dr.conf

	LVS_TYPE=VS_DR
	INITIAL_STATE=on 
	
	VIP=eth1:110 192.168.1.110 255.255.255.255 192.168.1.110
	DIRECTOR_INSIDEIP=eth0 director 192.168.1.0 255.255.255.0 192.168.1.255 
	DIRECTOR_DEFAULT_GW=client2
	SERVICE=t telnet rr 192.168.1.11 
	SERVICE=t telnet rr bashfull
	SERVICE=t netpipe rr 192.168.1.11 
	SERVICE=u nfs rr 192.168.1.11 
	SERVICE=t rsh rr sneezy 
	
	SERVER_VIP_DEVICE=lo:0
	SERVER_NET_DEVICE=eth0
	SERVER_DEFAULT_GW=client


lvs_nat.conf

	LVS_TYPE=VS_NAT
	INITIAL_STATE=on  
	VIP=eth1:110 lvs2 255.255.255.0 192.168.2.255
	DIRECTOR_INSIDEIP=eth0 director 192.168.1.0 255.255.255.0 192.168.1.255
	DIRECTOR_DEFAULT_GW=client2
	SERVICE=t telnet rr 192.168.1.11:telnet bashfull:telnet 
	SERVICE=t rsh rr sneezy:rsh 
	SERVICE=t netpipe rr 192.168.1.11:netpipe
	SERVICE=u nfs rr 192.168.1.11:nfs

	SERVER_NET_DEVICE=eth0
	SERVER_DEFAULT_GW=director_inside

=head1 BUGS

I haven't even thought about looking for bugs yet.

=head1 HISTORY

B<early version>, shell script Jul 99, for 2.0.36 kernels

B<later version>, perl script  Dec 99, for 2.2.x kernels, 
generates sh script output. Action is taken as conf file 
is read in. Requires entries to be in correct order.
Information from previous entries in conf file is discarded
so you can't take action based on two of the input lines.
Works OK enough.

B<v0.6 Jun 99>, perl script, entries in configure file 
are read in before any output is constructed. 
Entries in conf file can be in any order 
(this is not particularly useful) and (more importantly)
decisions can now be made using two (or more) of input data. 

=head1 TODO

1. merge configure-lvs with HA scripts (eg Horms stuff)

=head1 THANKS TO

discussions with Rob McCauley and Tommy Cathey at work.

=head1 LICENSE

configure-lvs.pl is distributed under the GPL. You should have
several hundred copies of it by now. If not please
contact
 
	Free Software Foundation, Inc.
	675 Mass Ave, Cambridge, MA 02139, USA

=head1 AUTHOR

This code is (C) Joseph Mack jmack@wm7d.net, Jun 00

This code is part of the Linux Virtual Server project
http://www.linuxvirtualserver.org/

=cut
#---------------------------------------------------------------------------------
#
# General setup
use strict;
#v0.4 17 Nov 99 fixed bug in ppc setup code
use Net::DNS;
use Socket;
$| = 1; #flush buffers

my ($lvs_type, $connection_parameter, $connection_type);
my $initial_state;
my $line; #input line

my @monitors = ("dialin", "dns", "fping", "freespace", "ftp", "ftp_kernelorg", "hpnp", "http", "http_t", "imap", "ldap", "msql-mysql", "na_quota", "netappfree", "nntp", "ping", "pop3", "process", "seq", "smtp", "ssh", "rd", "reboot", "tcp", "telnet", "up_rtt");
my $conf_file;
my $director_file = "rc.lvs_${lvs_type}director";
my $server_file = "rc.lvs_${lvs_type}server";
my $rc_file = "rc.lvs_$lvs_type";
my $mon_file = "mon_lvs${lvs_type}.cf";
my @params;
my ($vdevice,$vip, $vnetmask, $vbroadcast); #VIP setup for VS_TUN, VS_DR
my ($inside_device, $inside_ip, $inside_network, $inside_netmask, $inside_broadcast,$dot_quad, $classless, $log_2); #server network for VS_NAT

#service setup
my $default_scheduler = "wlc";
my ($proto, $service, $scheduler, $t_switch, $timeout, $server, $weight, $weight_lvsadm); 
my ($service_name, $port, $protocol, $server_name,$server_ip);
my ($server_port, $server_ipport, $server_service_name);
my (@service_lines, $element, $line_reference); #the input lines containing SERVICES, each element of the line, and the reference to the line inside @service_lines
my (@redirect_lines, @services);#for running rc.horms


#server setup
my ($rs_vip_device, $snetdevice, $server_counter, $monitor); 
my @servers; #list of servers

#kernel version
my $uname_r_known = "N";
my $uname_r_item; 
my $minor_rev;

#gateways, set to null incase no input data for them
my ($server_gw, $director_gw);
$server_gw="";
$director_gw="";

#other
my $temp;

my $line_counter;
$line_counter=0;
$server_counter=0;

#list of known schedulers (ppc is only supported to 2.2.10)
my @schedulers = ("rr","wrr","lc","wlc","pcc");

#list kernels that this script is thought to work on
#pre kernels are named like 2.2.15pre5. 
#this will be rev'ed down to 2.2.14.
my @uname_r_ok = ("2.2.10","2.2.12","2.2.13","2.2.14","2.2.15","2.2.16","2.4.0");
my $uname_r;

#------------------------------------------------------------
#shell boilerplate
#Perl Cookbook 1998, p23
my $initialise_variables;
($initialise_variables = <<'INITIAL') =~ s/^@//gm;
@MAINTAINER="jmack@wm7d.net"
@LVS_USERS="lvs-users@linuxvirtualserver.org"
@UNAME=`uname`
@UNAME_R=`uname -r`
@#initialise utilities. some of them will be in the path hopefully. 
@#At least this way a "file not found" error will result if they're in a different place.
@#Otherwise the "" command would have been issued with lots of arguements.
@#
@
@IFCONFIG="ifconfig"
@NETSTAT="netstat"
@ROUTE="route"
@AWK="awk"
@
@if [ `uname` = "SunOS" ]
@then
@	{
@	IFCONFIG="/usr/sbin/ifconfig"
@	NETSTAT="/usr/bin/netstat"
@	ROUTE="/usr/sbin/route"
@	AWK="/usr/bin/awk"
@	#echo "Solaris $IFCONFIG $NETSTAT $ROUTE"
@	}
@fi
@if [ `uname` = "Linux" ]
@then
@	{
@	IFCONFIG="/sbin/ifconfig"
@	NETSTAT="/bin/netstat"
@	ROUTE="/sbin/route"
@	AWK="/usr/bin/awk"
@	#echo "Linux $IFCONFIG $NETSTAT $ROUTE"
@	}
@fi
@IPVSADM="/sbin/ipvsadm"
@IPCHAINS="/sbin/ipchains"
@FPING="/usr/local/bin/fping"
INITIAL

my $ratz;
($ratz = <<'RATZ') =~ s/^@//gm;
@#
@#was rc.ratz--------------------------------------
@#
@#adds VIP on non-linux realservers operating in a VS-DR LVS
@#Info from Roberto Nibali (ratz) Ratz <ratz@tac.ch>
@#
@realserver_osversion_unknown()
@	{
@	echo "$0 error: unknown version ${UNAME_R} of OS $UNAME"
@	echo ""
@	echo "You may be able to configure the direct routing interface"
@	echo "on this realserver using the known commands for "
@	echo "$UNAME contained in this file."
@	echo " "
@	echo "if you are successfull, please send the output of the following commands"
@	echo ""
@	echo "uname "
@	echo "uname -r"
@	echo "the successful ifconfig command"
@	echo ""
@	echo "to $MAINTAINER or $LVS_USERS - thanks Joe"
@	exit 1
@	}
@
@realserver_os_unknown()
@	{
@	echo "$0 error: unknown OS $UNAME"
@	echo ""
@	echo "cannot configure non arping lo device"
@	echo " "
@	echo "if you can figure out how to do this, please send the output of the following commands"
@	echo ""
@	echo "uname "
@	echo "uname -r"
@	echo "the successful ifconfig command"
@	echo ""
@	echo "to $MAINTAINER or $LVS_USERS - thanks Joe"
@	exit 1
@	}
@
@install_realserver_vip()
@	{
@	echo "install_realserver_vip: configuring $UNAME $UNAME_R "
@	$IFCONFIG $DEV $ALIAS $VIP netmask $NETMASK -arp up 
@	echo "ifconfig output "
@	#$IFCONFIG -a
@	$IFCONFIG $DEV
@	echo "installing route for VIP $VIP on device $DEV"
@	$ROUTE add -host $VIP dev $DEV
@	echo "listing routing info for VIP $VIP "
@	$NETSTAT -rn | grep $VIP
@	}
@
@detect_realserver_os()
@	{
@	DEV=$SERVER_VIP_DEVICE
@	SHORT_DEV=${DEV%:*}	#lo:0 -> lo, tunl0 -> tunl0
@	#echo "SHORT_DEV=$SHORT_DEV"
@	case $UNAME in 
@		Linux )
@			case $UNAME_R in
@				2.0.* )
@				#DEV="lo:0"
@				NETMASK="0xffffffff"
@				ALIAS=""
@				install_realserver_vip
@				;;
@				2.2.* )
@				NETMASK="0xffffffff"
@				ALIAS=""
@				if [ "$SERVER_VIP_DEVICE" = "TP" ]
@				then
@					#don't need to detect OS version if installing TP
@					setup_realserver_TP
@				else
@					#test for "pre","-pre","-test" at the end of uname_r eg 2.2.15pre10
@					#you can't test for a string (like "pre") 
@					#but you can remove it and see if the string is changed.
@					#if [ ${UNAME_R%pre*} != "$UNAME_R" ]
@					#UNAME_R="2.2.1-9"	#for testing
@					#echo "UNAME_R $UNAME_R"		
@					#echo "UNAME_R no suffix ${UNAME_R%%[\-A-Za-z]*}"		
@					if [ ${UNAME_R%%[\-A-Za-z]*} != "$UNAME_R" ]
@					then
@						echo -n "OS verion ${UNAME}-${UNAME_R} being treated as minor version "
@						UNAME_R=${UNAME_R%%[\-A-Za-z]*}
@						MINOR_VERSION_NUMBER=${UNAME_R#2\.2\.}
@						#decrement MINOR_VERSION_NUMBER
@						if [ $MINOR_VERSION_NUMBER != "0" ]
@						then
@							MINOR_VERSION_NUMBER=$(($MINOR_VERSION_NUMBER - 1))
@						fi
@						echo "$MINOR_VERSION_NUMBER"
@						UNAME_R="2.2.${MINOR_VERSION_NUMBER}"
@					else
@						MINOR_VERSION_NUMBER=${UNAME_R#2\.2\.}
@					fi 
@					#
@                                       # echo "MINOR_VERSION_NUMBER $MINOR_VERSION_NUMBER"
@					if [ $CONNECTION_TYPE = $DR ] || [ $CONNECTION_TYPE = $TUN ]
@					then
@						#DEV="lo:0" or "tunl0"
@						install_realserver_vip
@						if [ $MINOR_VERSION_NUMBER -ge 14 ]
@						then
@							#hide VIP, OS minor_version >=14
@							if [ $HIDDEN = "Y" ]
@							then
@								echo "hiding interface $DEV, will not arp"
@								echo 1 > /proc/sys/net/ipv4/conf/all/hidden
@								echo 1 > /proc/sys/net/ipv4/conf/${SHORT_DEV}/hidden
@							else
@								echo "un-hiding interface $DEV, will arp"
@								echo 0 > /proc/sys/net/ipv4/conf/all/hidden
@								echo 0 > /proc/sys/net/ipv4/conf/${SHORT_DEV}/hidden
@							fi
@						else
@							echo "" 
@							echo "warning:${UNAME}-${UNAME_R} does not support hiding devices" 
@							echo "upgrade to at least 2.2.14 or else read the HOWTO for solving the arp problem"
@							echo "the VIP device on the realserver will arp and the LVS will not work in its current state"
@							echo "" 
@						fi 
@					else
@						echo "unknown connection type $CONNECTION_TYPE"
@					fi
@				fi
@				;;
@				* )
@				realserver_osversion_unknown
@				;;
@			esac
@			;;
@	
@	#uname      : FreeBSD
@	#uname -r   : 3.2-RELEASE
@	#<command>  : ifconfig lo0 alias <VIP> netmask 0xffffffff -arp up 
@	#ifconfig -a: lo0: flags=80c9<UP,LOOPBACK,RUNNING,NOARP,MULTICAST>mtu 16837
@	#                  inet 127.0.0.1 netmask 0xff000000
@	#                  inet <VIP> netmask 0xffffffff
@		FreeBSD )
@			case $UNAME_R in
@				3.2* | 3.3* | 2.2.5* )
@				#DEV="lo0"
@				NETMASK="0xffffffff"
@				ALIAS="alias"
@				if [ $CONNECTION_TYPE = $TUN ]
@				then
@					echo "error:tunneling not supported with ${UNAME}-${UNAME_R}"
@					echo "VIP device not installed, no action taken" 
@				else
@					install_realserver_vip
@				fi
@				;;
@				* )
@				realserver_osversion_unknown
@				;;
@			esac
@			;;
@	#
@	#uname      : IRIX
@	#uname -r   : 6.5
@	#<command>  : ifconfig lo0 alias <VIP> netmask 0xffffffff -arp up
@	#ifconfig -a: lo0: flags=18c9<UP,LOOPBACK,RUNNING,NOARP,MULTICAST,CKSUM>
@	#                  inet 127.0.0.1 netmask 0xff000000
@	#                  inet <VIP> netmask 0xffffffff
@		IRIX )
@			case $UNAME_R in
@				6.5 )
@				#DEV="lo0"
@				NETMASK="0xffffffff"
@				ALIAS="alias"
@				if [ $CONNECTION_TYPE = $TUN ]
@				then
@					echo "error:tunneling not supported with ${UNAME}-${UNAME_R}"
@					echo "VIP device not installed, no action taken" 
@				else
@					install_realserver_vip
@				fi
@				;;
@				* )
@				realserver_osversion_unknown
@				;;
@			esac
@			;;
@	
@	#uname      : SunOS
@	#uname -r   : 5.7
@	#<command>  : ifconfig lo0:1 <VIP> netmask 255.255.255.255 up
@	#ifconfig -a: lo0:  flags=849<UP,LOOPBACK,RUNNING,MULTICAST>mtu 8232
@	#                   inet 127.0.0.1 netmask ff000000
@	#             lo0:1 flags=849<UP,LOOPBACK,RUNNING,MULTICAST>mtu 8232
@	#                   inet <VIP> netmask ffffffff
@		SunOS )
@			case $UNAME_R in
@				5.7 )
@				#DEV="lo0"
@				NETMASK="0xffffffff"
@				ALIAS="alias"
@				if [ $CONNECTION_TYPE = $TUN ]
@				then
@					echo "error:tunneling not supported with ${UNAME}-${UNAME_R}"
@					echo "VIP device not installed, no action taken" 
@				else
@					install_realserver_vip
@				fi
@				;;
@				* )
@				realserver_osversion_unknown
@				;;
@			esac
@			;;
@	
@	#
@	#uname      : HP-UX
@	#uname -r   : B.11.00
@	#<command>  : ifconfig lan1:1 10.10.10.10 netmask 0xffffff00 -arp up
@	#ifconfig -a: lan0:   flags=842<BROADCAST,RUNNING,MULTICAST>
@	#                     inet <some IP> netmask ffffff00
@	#             lan0:1: flags=8c2<BROADCAST,RUNNING,NOARP,MULTICAST>
@	#                     inet <VIP> netmask ffffff00
@	#
@		HP-UX )
@			case $UNAME_R in
@			B.11.00 )
@				#DEV="lan1:1"
@				NETMASK="0xffffff00"
@				ALIAS=""
@				if [ $CONNECTION_TYPE = $TUN ]
@				then
@					echo "error:tunneling not supported with ${UNAME}-${UNAME_R}"
@					echo "VIP device not installed, no action taken" 
@				else
@					install_realserver_vip
@					echo "warning HP-UX lo replies to arp requests."
@					echo "This will work if the VIP is on a separate ethernet device."
@					echo "See the HOWTO about solutions to the arp problem."
@				fi
@				;;
@				* )
@				realserver_osversion_unknown
@				echo "error: HP-UX lo replies to arp requests."
@				echo "Cannot use with VS-DR"
@				;;
@			esac
@	
@		;;
@		* )
@		realserver_os_unknown
@		;;
@	esac
@	}
@#end rc.ratz-------------------------------------------
RATZ

my $horms;
($horms = <<'HORMS') =~ s/^@//gm;
@#--------was rc.horms--------------------------
@setup_realserver_TP()
@	{
@	#realserver and director TP setup are different
@	#on director accept all services
@	#on realserver only accept services destined for that realserver
@	#
@	#turn on IP forwarding (off by default in 2.2.x kernels)
@	echo "1" > /proc/sys/net/ipv4/ip_forward
@	
@	#flush ipchains table
@	$IPCHAINS -F input
@	
@	print_redirect_services
@	
@	#list ipchain rules
@	echo ""
@	echo "listing ipchains rules"
@	$IPCHAINS -L input
@	}
@#--------end rc.horms------------------------------------
@
HORMS

my $general_boilerplate;
($general_boilerplate = <<'BOILERPLATE') =~ s/^@//gm;
@
@remove_vip()
@	{
@	VIP_FOUND="N"
@	DEVICES=`$IFCONFIG -a | cut -c0-10 | xargs` 
@	for DEVICE in $DEVICES
@	do
@		#echo "checking device $DEVICE for VIP"
@		NUMBER_VIPS=`$IFCONFIG $DEVICE | grep -c $VIP`
@		if [ "${NUMBER_VIPS}" != "0" ]
@		then
@			#echo "device $DEVICE has VIP $VIP"
@			#if this is an ethernet aliase (eg eth0:110)
@			#then bringing it down will also bring down 
@			#all eth devices and 
@			#remove the entries from the routing table.
@			#
@			#However if we are just deinstalling and reinstalling
@			#the same aliased device (eg eth0:110) then we don't
@			#need to bother anyone with alarming messages.
@			#
@			if [ $DEVICE = $NEW_VIP_DEVICE ]
@			then
@				#do nothing, new device == old device
@				echo ""
@				#echo "old VIP device = new VIP device"
@			else
@				#do we have to remove an eth device?
@				SHORT_DEV=${DEVICE%:*}	#eth0:0 -> eth0, tunl0 -> tunl0
@				THIS_DEVICE=`expr "$DEVICE" : '\(...\).*' ` #first 3 chars of device, eth0:110 -> eth
@				#looking for eth devices only
@				if [ $THIS_DEVICE = "eth" ]
@				then
@					echo "For this configuration, the VIP $VIP on $DEVICE must be removed (not just down'ed)."
@					echo "The VIP $VIP is on an aliased/regular ethernet device $DEVICE."
@					echo "To just bring down $DEVICE will bring down all connections on $SHORT_DEV,"
@					echo "but $DEVICE would still be in the ifconfig table, "
@					echo "marked as down (or not marked UP)."
@					echo "The network connections to $SHORT_DEV and its aliases"
@					echo "would be gone from the routing table"
@					echo "and your $SHORT_DEV network will be hosed."
@					echo "If you are connected from a remote machine via $SHORT_DEV,"
@					echo "then your connection would be lost."
@					echo ""
@					echo "This script will exit here to prevent loosing editing sessions etc."
@					echo "You should re-execute this script after removing the $DEVICE entry"
@					echo "from the ifconfig table (not just downing it)."
@					echo "If you were to down $DEVICE, then when any IP on $SHORT_DEV is up'ed," 
@					echo "then $DEVICE will also be up'ed, so down'ing an aliased device doesn't remove it."
@					echo "Unlike Solaris you can't unplumb $SHORT_DEV." 
@					echo "You can reboot :-(, or unload the driver for $SHORT_DEV."
@					echo "To unload the module without dropping connections,"
@					echo "make sure the connections are idle (you'll loose the connections otherwise), "
@					echo "then at the console do"
@					echo ""
@					echo "# ifconfig eth0 down;ifconfig eth1 down (all eth devices);\\"
@					echo "   rmmod tulip (your ethernet module);\\"
@					echo "   . /etc/rc.d/rc.inet1 (or whatever brings up your network)"
@					echo ""
@					echo "If you remove $DEVICE successfully, "
@					echo "you will not see this notice next time you run this script."
@					exit 1
@				fi # eth device   
@			fi # replacing VIP onto device different to that found with ifconfig -a
@			#echo "is it up?"
@			if [ `$IFCONFIG $DEVICE | grep -c "UP"` != "0" ]
@			then
@				#echo "it's up"
@				VIP_FOUND="Y"
@				#echo "device $DEVICE has $VIP and is UP"
@				#echo "removing $VIP from $DEVICE"
@				$IFCONFIG $DEVICE down
@			else
@				#echo "it's not up"
@				echo -n ""
@			fi 
@		fi
@	done
@
@	DEVICES=`$NETSTAT -rn | grep $VIP | awk '{print $8}' | xargs` 
@	for DEVICE in $DEVICES
@	do
@		#echo "removing route to VIP through $DEVICE"
@		route del -host $VIP $DEVICE
@	done
@
@	if [ $VIP_FOUND = "N" ]
@	then
@		#echo "$VIP not found on any network devices, good"
@		echo -n ""
@	fi
@	echo ""
@	}
@
@get_current_gw(){
@	# routing table can have multiple default gw
@	NUMBER_DEFAULT_GW=`netstat -rn| grep "^0.0.0.0" |awk '{print $2}' |wc -l`
@	#get first default gw
@	CURRENT_GW=`$NETSTAT -rn| grep "^0.0.0.0" |awk '{print $2}'| xargs | awk '{print $1}'`
@	#echo "number of default gw $NUMBER_DEFAULT_GW, first gw $CURRENT_GW "
@	}
@	
@install_default_gw(){
@	get_current_gw
@	
@	#it's a little tricky to test if there is only one default gw 
@	#and it's the right IP. 
@	#just strip out all the gw's and re-install the correct one
@	#
@	echo " "
@	while [ $NUMBER_DEFAULT_GW != "0" ]
@	do
@		#echo "deleting current default gw $CURRENT_GW"
@		$ROUTE del default gw $CURRENT_GW
@		get_current_gw
@	done
@	
@	echo "changing default gw to $DEFAULT_GW"
@	route add default gw $DEFAULT_GW
@	echo "showing routing table"
@	$NETSTAT -rn
@	echo ""
@	echo -n "checking if DEFAULT_GW $DEFAULT_GW is reachable - "
@	$FPING $DEFAULT_GW | tr -d '\n'	#remove \n from fping
@	if [ $? = "0" ]
@	then 
@		echo ", good"
@	else
@		echo ", warning: this machine must be able to connect to the default gw for the LVS to work"
@	fi
@	}
@
@#--------end boilerplate------------------------------------
@
@check_for_DIIP(){
@	#since this is a realserver, there should be no DIIP here 
@	#here and it should have been installed on the director by now. 
@	echo "looking for DIIP $DIRECTOR_INSIDEIP "
@	$FPING $DIRECTOR_INSIDEIP 
@	if [ $? = "0" ] 
@	then 
@		echo "found, good"
@		#check that it's not local.
@		#(we shouldn\'t have got here if it is)
@		if [ `$IFCONFIG -a | grep -c "$DIRECTOR_INSIDEIP "` = "0" ] 
@		then 
@			echo "not local, good "
@		else 
@			echo "error: DIIP is local - "
@			echo "we shouldn't even have got here!"
@		fi 
@	 else 
@	 	echo "DIIP not found, director has not been setup yet." 
@	 	echo "rerun this script after setting up director" 
@	 	echo "so that all tests can be run." 
@	 fi 
@	 } #check_for_DIIP 
@
@check_for_vip_on_director(){
@	echo ""
@	#echo "looking for VIP on director and/or realserver from realserver"
@	#there is no VIP device installed yet on realserver
@	#VS-NAT will not get a VIP device on the realserver
@	echo ""
@	echo "looking for VIP on director from realserver"
@	if [ $DIRECTOR_VIP_DEVICE = "TP" ] 
@	then
@		echo "director is accepting packets by TP (no VIP on director)" 
@	 	echo "bypassing ping check of VIP $VIP" 
@	else
@	 	echo "director is accepting packets on network device $DIRECTOR_VIP_DEVICE" 
@		echo "pinging VIP " 
@		$FPING $VIP	
@		if [ "$?" = "0" ] 
@		then 
@			echo "$VIP found, good. It's not on this server, assume it's on the director. "
@		else 
@			echo "warning:$VIP not found, presumably director has not been set up yet"
@			echo "rerun this script after setting up the director if you want all checks to be positive"
@		fi 
@	fi
@	} #check_for_vip_on_director 
@
@#--------end boilerplate------------------------------------
BOILERPLATE

#------------------------------------------------------------

sub calc_log_2
	{
	$log_2 = 0;
	my $temp = $_[0];
	#print "$temp \n";
	while ($temp != 1)
		{
		$temp = $temp >> 1;
		$log_2++;
		#print "$temp $log_2 \n";
		}
	#print "$log_2 \n";
	return ($log_2);
	}

sub dotquad_to_classless
	{
	#takes 255.255.255.0, converts it to 24
	$dot_quad = $_[0];
	#print "dot_quad: $dot_quad \n";
	$dot_quad =~ /(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})/ ;
	#print "$1 $2 $3 $4 \n";
	$classless = calc_log_2(eval($1+1)) + calc_log_2(eval($2+1)) + calc_log_2(eval($3+1)) + calc_log_2(eval($4+1)); 
	#print "classless $classless \n";
	return ($classless);
	}

sub parse_hostname
	{
	my $host = $_[0];
	if ($host =~ /\s/) {print "parse_hostname: error - null hostname or ip \n";}
	#print "parse_hostname: $host\n";
	my ($host_name, $host_ip, $packed_ip); 

	#host can be a host_ip (eg 192.168.1.19) or a name in /etc/hosts (eg grumpy, grumpy.mack.net)
	#whichever version is given, we need to find the other version too

	#if $host contains letters, then assume it is a name and find corresponding host_ip 
	if ($host =~ /\d{1,3}.\d{1,3}.\d{1,3}.\d{1,3}/)	
		{
		#$host is a host_ip number and not a host name
		$host_ip = $host;
		$host_name = gethostbyaddr(inet_aton("$host_ip"),AF_INET);
		if ($host_name eq '') 
			{
			#accept and continue
			#print "parse_hostname: no name available for host on host_ip $host_ip, will use host_ip number instead. \n";
			#print "you could enter a name in /etc/hosts file for the host on host_ip $host_ip \n";
			#print "and this message will go away. \n";
			$host_name = $host;
			}
		else
			{
			#print "parse_hostname: host for host_ip $host is $host_name \n";
			}
		}
	else
		{
		#$host is a name
		$host_name = $host;
		$packed_ip = gethostbyname($host_name) or die "parse_hostname: error: gethostbyname failure - Couldn't lookup hostname $host_name : $! \n";
		$host_ip = inet_ntoa($packed_ip);
		if ($host_ip eq '')
			{
			print "parse_hostname: host $host_name does not resolve to a valid host_ip number \n";
			exit 1;
			}
		else
			{
			#print "parse_hostname: host_ip for host $host_name is $host_ip \n";
			}
		}
	return ($host_name, $host_ip);
	}

sub parse_service
	{
	my $service = $_[0];	
	my ($service_name, $port); 

	#service can be a port (eg 80) or a name in /etc/services (eg http)
	#whichever version is given, we need to find the other version too

	#if $service contains no digits, then assume it is a name and find corresponding port 
	if ($service =~ /\D/)	
		{
		#$service is a name
		$service_name = $service;
		$port = getservbyname($service, $protocol);
		if ($port eq '')
			{
			print "service $service_name does not give a valid port number in /etc/services \n";
			exit 1;
			}
		else
			{
			#print "port for service $service_name is $port \n";
			}
		}
	else 	
		{
		#$service is a port number and not a service name
		$port = $service;
		$service_name = getservbyport($service,$protocol);
		if ($service_name eq '') 
			{
			#accept and continue
			print "no name available for service on port $port, will use port number instead. \n";
			print "you could enter a name in /etc/services file for the service on port $port \n";
			print "and this message will go away. \n";
			$service_name = $service;
			}
		else
			{
			#print "service for port $service is $service_name \n";
			}
		}
	return ($service_name, $port);
	}

sub set_monitor
	{
	#does this service have a monitor? if not set fping as the default monitor
	$monitor = "fping";
	foreach $temp (@monitors)
		{
		if ($temp eq $service_name)
			{
			$monitor = $temp;
			}
		}			
	}
		
sub write_mon_hostgroup
	{
	print MON_FILE "#------\n";
	print MON_FILE "hostgroup LVS$server_counter $server_ip \n";
	print MON_FILE "\n"; 
	print MON_FILE "watch LVS$server_counter \n";
	print MON_FILE "#the string/text following service (to OEL) is put in header of mail messages\n"; 
	#print MON_FILE "#The mon docs imply that a string is required (ie no blanks), but the test.alert and mail.alert handle text (several words) just fine.\n"; 
	#print MON_FILE "#virtualserver.alert bombs if text is sent (appears to be multiple parameters to Getopt), so the text is encased in \"\".\n"; 
	print MON_FILE "service \"$service_name on LVS$server_counter $server_ip\" \n";
	print MON_FILE "	interval 15s \n";
	if ($monitor eq "dns")
		{
		#the dns.monitor needs to know a zone and the master DNS machine for that zone.
		#the monitor checks the master for the SOA/serial number of the zone
		#and then requests the same info from the machine being monitored (here the realserver)
		#Since any zone will do, we will monitor the domain that the LVS machine belongs to.
		#Since the only domain the script knows about is the domain of the LVS, the script
		#will look for the LVS domain in /etc/resolv.conf
		#The director will be the master machine for the domain, whose only function need be to serve up
		#a serial number for the domain and the IP/name of the director machine.
		#The realservers need only be caching secondaries for the LVS domain. 
		#For real queries they will have to go out onto the net anyhow.
		#Once the LVS is running, if the files on the master machine are updated, 
		#the serial number on the copy on the realservers will not be updated till the 
		#timeout/refresh period (usually several hours).
		#In this case named on the realservers will have to be shutdown, 
		#files used as cache backup deleted and then named on the realservers restarted 
		#

		#look for domain in resolv.conf, then look for SOA nameserver for domain
		my $resolv_conf_file = "/etc/resolv.conf";
		open (RESOLV_CONF_FILE, "<$resolv_conf_file")|| die "Can't read $resolv_conf_file $!\n";
		my $domain;
		my $ns;
		my $line_counter=0;
		while (<RESOLV_CONF_FILE>)
			{
			$line_counter++;
			chomp;
			#print "resolv.conf $_\n";
			if ($_ =~ /^domain (.*) */)
				{
				$domain = $1;
				#print "domain: $domain $line_counter\n";
				} 
			}
		#find the nameservers for $domain
		#code modelled on dns.monitor (C) 1998 David Eckelkamp
		#and on Net::DNS man page
		my $res = new Net::DNS::Resolver;
		my $query = $res->query ($domain, "SOA");
		if ($query)
			{
			#($query->answer)[0] ->print;
			unless ($query->header->aa)
				{
				#print "nameserver not authoritative for $domain \n";
				}
			$ns = ($query->answer)[0]->string;
			#print "ns $ns\n";
			#non-greedy matching	
			if ($ns =~ /SOA\s+(.*?)\.\s+/)
				{
				#print "chopped ns $1 \n";
				$ns = $1;
				print MON_FILE "	monitor $monitor.monitor -zone $domain -master $ns \n";
				}
			}
		else
			{
			print "search for SOA for $domain failed: ", $res->errorstring, "\n";
			exit 1;
			}

		#search for nameservers (is SOA's)
		#my $query = $res->query($domain, "NS");
		#my $rr;
		#if ($query)
		#	{
		#	foreach $rr ($query->answer)
		#		{
		#		next unless $rr->type eq "NS";
		#		print $rr->nsdname, "\n";
		#		}
		#	}
		#else
		#	{
		#	print "search for nameserver for $domain failed: ", $res->errorstring, "\n";
		#	}
#		print MON_FILE "	monitor $monitor.monitor -zone mack.net -master di.mack.net \n";
		}
	else
		{
		print MON_FILE "	monitor $monitor.monitor \n";
		}
	print MON_FILE "	allow_empty_group \n";
	print MON_FILE "	period wd {Sun-Sat} \n";
	print MON_FILE "		alert mail.alert root \n";
	print MON_FILE "		upalert mail.alert root \n";
	print MON_FILE "		#-V is virtual service, -R is remote service, -P is protocol, -A is add/delete (t|u)\n";
	if ($lvs_type =~ /tun|dr/)
		{
		print MON_FILE "		alert virtualserver.alert -A -d -P -$proto -V $vip:$port -R $server_ip \n";
		#print MON_FILE "		upalert virtualserver.alert -A -a -P -$proto -V $vip:$port -R $server_ip -T $connection_parameter -w $weight\n";
		print MON_FILE "		upalert virtualserver.alert -A -a -P -$proto -V $vip:$port -R $server_ip -T \$CONNECTION_TYPE -w $weight\n";
		}
	elsif ($lvs_type eq "nat")
		{
		print MON_FILE "		alert virtualserver.alert -A -d -P -$proto -V $inside_ip:$port -R $server_ip:{$server_port} \n";
		#print MON_FILE "		upalert virtualserver.alert -A -a -P -$proto -V $inside_ip:$port -R ${server_ip}:${server_port} -T $connection_parameter -w $weight\n";
		print MON_FILE "		upalert virtualserver.alert -A -a -P -$proto -V $inside_ip:$port -R ${server_ip}:${server_port} -T \$CONNECTION_TYPE -w $weight\n";
		}
	else	
		{
		print "error: unknown LVS_type $lvs_type \n";
		}

	print MON_FILE "\n";
	print MON_FILE "#the line above must be blank\n";
	print MON_FILE "\n";
	}

sub open_files
	{
	$rc_file = "rc.lvs_$lvs_type";
	$mon_file = "mon_lvs${lvs_type}.cf";
	open (RC_FILE, ">$rc_file")|| die "Can't write to $rc_file $!\n";
	open (MON_FILE, ">$mon_file")|| die "Can't write to $mon_file $!\n";
	}

sub print_headers
	{
	print RC_FILE "#!/bin/bash \n";
	print RC_FILE "#rc.lvs_${lvs_type} (C) Joseph Mack 1999,2000.\n"; 
	print RC_FILE "#autogenerated by configure_lvs.pl from lvs_${lvs_type}.conf.\n";
	print RC_FILE "#Feel free to edit.\n";
	print RC_FILE "#This file needs a version of sh that has subroutines.\n";
	print RC_FILE "#sh on Solaris 2.7 and bash on Linux are fine.\n";
	print RC_FILE "#\n";
	print RC_FILE "#Setup for a Linux Virtual Server (LVS) running in $lvs_type mode.\n";
	print RC_FILE "#This code is part of the LVS project http://www.linuxvirtualserver.org\n";
	print RC_FILE "#The rc.lvs files produced by this code runs on Solaris machines, \n";
	print RC_FILE "#but I haven't been able to test whether it actually sets up a working realserver yet\n";
	print RC_FILE "#\n";
	print RC_FILE "#---begin boilerplate--------------------------------------------------------\n";
	print RC_FILE "#\n";
	print RC_FILE "$ratz";
	print RC_FILE "$horms";
	print RC_FILE "$general_boilerplate";

	print MON_FILE "#mon_lvs${lvs_type}.cf \n";
	print MON_FILE "\n";
	print MON_FILE "#an example mon.cf file for a group of LVS machines running in $lvs_type mode\n";
	print MON_FILE "#generated by configure_lvs.pl - Feel free to edit\n";
	print MON_FILE "#This code is part of the LVS project http://www.linuxvirtualserver.org/\n";
	print MON_FILE "#for more information about LVS please see the LVS webpage\n";
	print MON_FILE " \n";
	print MON_FILE "#adapted from the mon.cf script supplied by Jim Trocki as part of the mon distribution.\n";
	print MON_FILE "#for info about mon see http://www.kernel.org/software/mon/\n";
	print MON_FILE " \n";
	print MON_FILE "#<IMPORTANT NOTE FROM Jim Trocki>\n";
	print MON_FILE "#\n";
	print MON_FILE "#NB:  hostgroup and watch entries are terminated with a blank line,\n";
	print MON_FILE "#or end of file. Don\'t forget the blank lines between them or you lose.\n";
	print MON_FILE "#\n";
	print MON_FILE "#</IMPORTANT NOTE FROM Jim Trocki>\n";
	print MON_FILE "#------------------------------------------------------------------\n";
	print MON_FILE "#\n";
	print MON_FILE "#mon config info, you probably don\'t need to change this very much\n";
	print MON_FILE "#\n";
	print MON_FILE "\n";
	print MON_FILE "alertdir   = /usr/lib/mon/alert.d \n";
	print MON_FILE "mondir     = /usr/lib/mon/mon.d \n";
	print MON_FILE "#maxprocs    = 20\n";
	print MON_FILE "histlength = 100\n";
	print MON_FILE "#delay before starting\n";
	print MON_FILE "randstart = 60s\n";
	print MON_FILE "\n";
	} 

sub print_close_mon_file
	{
	print MON_FILE      "#----mon_lvs${lvs_type}.cf------------------------------------------------------------------\n";
	}

sub setup_director_vip
	{
	print RC_FILE "	#setup VIP\n";
	print RC_FILE "	echo \"adding ethernet device and routing for VIP \$VIP \"\n";
	#wierd. the next command worked for months, then failed
	#print RC_FILE "	\$IFCONFIG \$DIRECTOR_VIP_DEVICE \$VIP netmask $vnetmask broadcast $vbroadcast up \n";
	print RC_FILE "	\$IFCONFIG \$DIRECTOR_VIP_DEVICE \$VIP netmask $vnetmask \n";
	print RC_FILE "	\$ROUTE add -host \$VIP dev \$DIRECTOR_VIP_DEVICE \n";
	print RC_FILE "	echo \"listing ifconfig info for VIP \$VIP \"\n";
	print RC_FILE "	\$IFCONFIG \$DIRECTOR_VIP_DEVICE \n";
	print RC_FILE "	echo -n \"checking VIP \$VIP is reachable from self (director): \"\n";
	print RC_FILE "	\$FPING \$VIP \n";
	print RC_FILE "	echo \"listing routing info for VIP \$VIP \"\n";
	print RC_FILE "	\$NETSTAT -rn | grep \$VIP \n";
	print RC_FILE "	echo \"\"\n";
	print RC_FILE "\n";
	}

sub open_setup_director
	{
	print RC_FILE "\n";
	print RC_FILE "setup_director() { \n";
	print RC_FILE "	echo \"LVS director \"\n";
	print RC_FILE "	echo \"\"\n";
	print RC_FILE "	echo \"clearing ipchain rules \"\n";
	print RC_FILE "	\$IPCHAINS -F \n";
	print RC_FILE "	#echo \"displaying ipchain rules (should be empty) \"\n";
	print RC_FILE "	#\$IPCHAINS -L \n";
	print RC_FILE "	#echo \"\"\n";
	print RC_FILE "	echo \"turn on ip forwarding\"\n";
	print RC_FILE "	echo \"1\" >/proc/sys/net/ipv4/ip_forward \n";
	if (!($lvs_type =~ /nat/))
		{
		#turn off icmp redirects for VS-DR, VS-TUN, for kernels =<2.2.12
		$uname_r =~ /^2\.2\.(\d+)/ ;
		my $kernel_minor_version = $1;
		#print "kernel minor version $kernel_minor_version \n";
		if ($kernel_minor_version < 13)
			{
			print RC_FILE "	echo \"kernel version $uname_r - turn off icmp redirects\"\n";
			print RC_FILE "	echo \"0\" >/proc/sys/net/ipv4/conf/eth0/send_redirects \n";
			}
		}
	print RC_FILE "	echo \"\"\n";
	print RC_FILE "	#don't bother removing device with VIP if it's going to be the replacement \n";
	print RC_FILE "	NEW_VIP_DEVICE=\$DIRECTOR_VIP_DEVICE \n";
	print RC_FILE "	remove_vip \n";
	print RC_FILE "	NEW_VIP_DEVICE=\"\" \n";
	print RC_FILE " \n";
	}

sub setup_director_TP
	{
	#print "setup_director_TP: start \n";
	print RC_FILE "	#setup accepting connection by TP\n";
	print RC_FILE "	echo \" \"\n";
	print RC_FILE "	echo \"Director is accepting packets by transparent proxy (TP).\"\n";
	print RC_FILE "	echo \"When using TP, the director does not have an IP for the VIP.\"\n";
	print RC_FILE "	echo \"You must install a route on the router/test client telling it how to get to the VIP.\"\n";
	print RC_FILE "	echo \"You can put a host route to an IP already on the director \"\n";
	print RC_FILE "	echo \"eg route add -host \$VIP gw director \"\n";
	print RC_FILE "	echo \"or \"\n";
	print RC_FILE "	echo \"put a (permanent) arp entry (arp -f /etc/ethers or arp -s) pointing to a MAC address on the director.\"\n";
	print RC_FILE "	echo \"\"\n";
	print RC_FILE "	echo \"Installing director redirect rules...\"\n";
	foreach $line_reference (@redirect_lines)
		{
		($server_ip, $service_name, $protocol) = @$line_reference; 
		#print "setup_director_TP: $server_ip, $service_name, $protocol \n";
		print RC_FILE " 	echo \"redirecting \${VIP}:$service_name:$protocol 	to local:$service_name \" \n";
		print RC_FILE " 	\$IPCHAINS -A input -j REDIRECT $service_name -d \$VIP $service_name -p $protocol \n";
		print RC_FILE " 	if [ \$? != \"0\" ] \n";
		print RC_FILE " 	then \n";
		print RC_FILE " 		echo \"setup_director_TP: error - transparent proxy not installed?\" \n";
		print RC_FILE " 	fi\n";
		print RC_FILE "	\n";
		}
	print RC_FILE "	\n";
	#print "\n";
	}

sub setup_director_masq_table
	{
	#my ($server_name, $server_ip);
	print RC_FILE "\n";
	if ($lvs_type =~ /nat/)
		{
		#used to masquerade all IPs, all ports
		#now just masquerade IP:services being LVS'ed
		#my $classless;
		#$classless = dotquad_to_classless($inside_netmask);
		#print RC_FILE "	\$IPCHAINS -A forward -j MASQ -s ${inside_network}/${classless} -d 0.0.0.0/0\n";
		print RC_FILE " ";	
		print RC_FILE "	echo \"installing ipchain rules\"\n";
		foreach $line_reference (@redirect_lines)
			{
			($server_name, $service_name, $protocol) = @$line_reference; 
			($server_name, $server_ip) = parse_hostname($server_name);
			print "setup_director_masquerade: $server_name, $service_name, $protocol \n";
			#print "setup_director_masquerade: $server_ip, $service_name, $protocol \n";
			#print RC_FILE " 	echo \"masquerading $protocol $server_ip:$service_name:$protocol to outside world\" \n";
			print RC_FILE " 	echo \"masquerading $protocol $server_name:$service_name:$protocol to outside world\" \n";
			#print RC_FILE " 	\$IPCHAINS -A forward -p $protocol -j MASQ -s $server_ip $service_name -d 0.0.0.0/0\n";
			print RC_FILE " 	\$IPCHAINS -A forward -p $protocol -j MASQ -s $server_name $service_name -d 0.0.0.0/0\n";
			}
		print RC_FILE "	\n";
		print RC_FILE "	echo \"ipchain rules \"\n";
		print RC_FILE "	\$IPCHAINS -L\n";
		print RC_FILE "\n";
		}
	}

sub setup_ipvsadm_table
	{
	print RC_FILE "	echo \" \"\n";
	print RC_FILE "	#echo \"clearing ipvsadm table \"\n";
	print RC_FILE "	\$IPVSADM -C \n";
	print RC_FILE "	echo \"installing LVS services with ipvsadm\"\n";
	print RC_FILE "	print_ipvsadm_lines \n";
	print RC_FILE "	#display current settings\n";
	print RC_FILE "	echo \" \"\n";
	print RC_FILE "	echo \"displaying ipvsadm settings \"\n";
	print RC_FILE "	\$IPVSADM \n";
	print RC_FILE "	echo \"\"\n";
	} 

sub setup_director_insideip
	{
	my $director_inside_ip_name;
	($director_inside_ip_name, $inside_ip) = parse_hostname($inside_ip);
	print RC_FILE "	#setup DIRECTOR_INSIDEIP\n";
	#print RC_FILE "	echo \"adding ethernet device and routing for DIRECTOR_INSIDEIP \$DIRECTOR_INSIDEIP \"\n";
	print RC_FILE "	echo \"adding ethernet device and routing for DIRECTOR_INSIDEIP $director_inside_ip_name \"\n";
	print RC_FILE "	\$IFCONFIG $inside_device $inside_ip netmask $inside_netmask broadcast $inside_broadcast up \n";
	print RC_FILE "	\$ROUTE add -net $inside_network netmask $inside_netmask dev $inside_device \n";
	print RC_FILE "	echo -n \"checking if director INSIDEIP $director_inside_ip_name is reachable from director: \"\n";
	print RC_FILE "	\$FPING $director_inside_ip_name \n";
	} #setup_director_insideip

sub print_close_setup_director
	{
	print RC_FILE "	echo \"\"\n";
	#print RC_FILE "	if [ \"$director_gw\" = \"\" ] \n";
	print RC_FILE "	if [ \$DIRECTOR_DEFAULT_GW = \"\" ] \n";
	print RC_FILE "	then \n";
	print RC_FILE "		echo \"no default gw set in input data. \"\n";
	print RC_FILE "		echo \"make sure each director has a route to the outside world/client(s) \" \n";
	print RC_FILE "		echo \"This will normally be setup by rc.inet1 \" \n";
	print RC_FILE "	else \n";
	#print RC_FILE "		DEFAULT_GW=\"$director_gw\" \n";
	print RC_FILE "		DEFAULT_GW=\$DIRECTOR_DEFAULT_GW \n";
	print RC_FILE "		install_default_gw \n";
	print RC_FILE "	fi \n";
	print RC_FILE "\n";
	print RC_FILE "	echo \" \" \n";
	print RC_FILE "	echo \"checking for valid server_gw \"\n";
	print RC_FILE "	#note blank at the end of the IP \n";
	#print "lvs_type $lvs_type \n";
	if ($lvs_type eq "nat")
		{
		#print RC_FILE "	if [ `\$IFCONFIG -a | grep -c \"$server_gw \"` = \"1\" ] \n";
		print RC_FILE "	if [ `\${IFCONFIG} -a | grep -c \"\${SERVER_DEFAULT_GW} \"` = \"1\" ] \n";
		print RC_FILE "	then \n";
		#print RC_FILE "		echo \"default gw for vs-$lvs_type servers is on director, good \" \n";
		print RC_FILE "		echo \"default gw for the \$LVS_TYPE servers is on director, good \" \n";
		print RC_FILE "	else \n";
		#print RC_FILE "		echo \"warning: the default gw for the vs-$lvs_type servers is NOT on the director \" \n";
		print RC_FILE "		echo \"warning: the default gw for the \$LVS_TYPE servers is NOT on the director \" \n";
		print RC_FILE "		echo \"you probably do not want this - a standard vs-$lvs_type LVS will not work this way. \" \n";
		print RC_FILE "	fi \n";
		}
	if ($lvs_type eq "dr")
		{
		print RC_FILE "	if [ `\$IFCONFIG -a | grep -c \"$server_gw \"` = \"0\" ] \n";
		print RC_FILE "	then \n";
		#print RC_FILE "		echo \"default gw for vs-$lvs_type servers is NOT on director, good \" \n";
		print RC_FILE "		echo \"default gw for \$LVS_TYPE servers is NOT on director, good \" \n";
		print RC_FILE "	else \n";
		#print RC_FILE "		echo \"warning: the default gw for the vs-$lvs_type servers is on the director \" \n";
		print RC_FILE "		echo \"warning: the default gw for the \$LVS_TYPE servers is on the director \" \n";
		#print RC_FILE "		echo \"you probably do not want this - a standard vs-$lvs_type LVS will not work this way. \" \n";
		print RC_FILE "		echo \"you probably do not want this - a standard \$LVS_TYPE LVS will not work this way. \" \n";
		print RC_FILE "		echo \"this setup will need Julian's martian modification to the kernel to work \" \n";
		print RC_FILE "	fi \n";
		}
	if ($lvs_type eq "tun")
		{
		#separate code for tun incase we start using this in a different manner
		print RC_FILE "	if [ `\$IFCONFIG -a | grep -c \"$server_gw \"` = \"0\" ] \n";
		print RC_FILE "	then \n";
		print RC_FILE "		echo \"default gw for vs-$lvs_type servers is NOT on director, good \" \n";
		print RC_FILE "	else \n";
		#print RC_FILE "		echo \"warning: the default gw for the vs-$lvs_type servers is on the director \" \n";
		print RC_FILE "		echo \"warning: the default gw for the \$LVS_TYPE servers is on the director \" \n";
		#print RC_FILE "		echo \"you probably do not want this - a standard vs-$lvs_type LVS will not work this way. \" \n";
		print RC_FILE "		echo \"you probably do not want this - a standard \$LVS_TYPE LVS will not work this way. \" \n";
		print RC_FILE "		echo \"this setup will need Julian's martian modification to the kernel to work \" \n";
		print RC_FILE "	fi \n";

		}
	print RC_FILE "	echo \"\"\n";
	print RC_FILE "	} #setup_director\n";
	print RC_FILE "\n";
	} #print_close_setup_director

sub install_realserver_vip_device
		{
		print RC_FILE "	#install non-arping network device on Server\n";
		print RC_FILE "	#don't bother removing device with VIP if it's going to be the replacement \n";
		print RC_FILE "	NEW_VIP_DEVICE=\$SERVER_VIP_DEVICE \n";
		print RC_FILE "	remove_vip \n";
		print RC_FILE "	NEW_VIP_DEVICE=\"\" \n";
		print RC_FILE "	detect_realserver_os \n";
		print RC_FILE "	echo \" \"\n";
		}

sub install_realserver_gw
	{
	print RC_FILE "	echo \"\"\n";
#	print "lvs_type $lvs_type \n";
#	default gw set in main:
#	if ($lvs_type eq "nat")
#		{
#		print RC_FILE "		echo \"installing director\'s inside_ip as default gw for $lvs_type realserver \" \n";
#		$server_gw = $inside_ip;
#		print "server_gw $server_gw \n";
#		}
#	else
#		{
#		print RC_FILE "		echo \"installing server_gw $server_gw as default gw for $lvs_type realserver \" \n";
#		print "server_gw not nat $server_gw \n";
#		}
#	print RC_FILE "	if [ \"$server_gw\" = \"\" ] \n";
	print RC_FILE "	if [ \$SERVER_DEFAULT_GW = \"\" ] \n";
	print RC_FILE " then \n";
	print RC_FILE "		echo \"warning: no default gw set in input data. \"\n";
	print RC_FILE "		echo \"make sure each server has a route to the outside world/client(s) \" \n";
	print RC_FILE "		echo \"This will normally be setup by rc.inet1 \" \n";
	print RC_FILE "	else \n";
	#print RC_FILE "		DEFAULT_GW=\"$server_gw\" \n";
	print RC_FILE "		DEFAULT_GW=\$SERVER_DEFAULT_GW \n";
	print RC_FILE "		install_default_gw \n";
	print RC_FILE "	fi \n";
	print RC_FILE "\n";
	}

sub check_default_route_nat
	{
	print RC_FILE "	echo \"checking default routing for \$LVS_TYPE realserver \" \n";
	print RC_FILE "	echo \"packets to director\'s default gw should go through director.\" \n";
	print RC_FILE "	echo \"(this test will return quickly if the routing is wrong for VS-NAT,)\" \n";
	print RC_FILE "	echo \"(will return in about 2 secs if setup correctly,)\" \n";
	print RC_FILE "	echo \"(and will hang if the routing is deranged.)\" \n";
	print RC_FILE "	echo \"Is director\'s default gw 2 hops away and is director one hop away on the path to the director\'s gw?\" \n";
	print RC_FILE "	# do this once for icmp redirects to reroute packets (if they are going to)\n";
	print RC_FILE "	traceroute $director_gw > /dev/null 2>&1 \n";
	print RC_FILE "	# do it for real \n";
	print RC_FILE "	HOPS_TO_DIRECTORS_GW=`traceroute $director_gw 2>/dev/null | wc -l` \n";
	print RC_FILE "	HOPS_TO_DIRECTOR=`traceroute $director_gw 2>/dev/null | grep \"($server_gw)\" | cut -c0-4` \n"; 
	print RC_FILE "	if [ \$HOPS_TO_DIRECTORS_GW = \"2\" ] && [ \$HOPS_TO_DIRECTOR = \"1\" ] \n";
	print RC_FILE "	then \n";
	print RC_FILE "		echo \"yes: good\" \n";
	print RC_FILE "	else \n";
	print RC_FILE "	 	echo \"error: the path to the director\'s default gw does not go through the director. \" \n";
	print RC_FILE "		echo \"hops to director\'s gw \$HOPS_TO_DIRECTORS_GW \" \n";
	print RC_FILE "		echo \"hops to director \$HOPS_TO_DIRECTOR \" \n";
	print RC_FILE "		echo \"this vs-$lvs_type LVS will not work.\" \n";
	print RC_FILE "		echo \"you can fix this by changing the IP's, networks and routing of the LVS.\" \n";
	print RC_FILE "		echo \"1. the network for the realservers must be private.\" \n";
	print RC_FILE "		echo \"2. the default gw for the realservers must be the director.\" \n";
	print RC_FILE "		echo \"3. a route to the director is not good enough, it won\'t work, the director must be the default gw.\" \n";
	print RC_FILE "		echo \"4. the realservers must not have any other routes to the client.\" \n";
	print RC_FILE "		echo \"(Some routing problems are fixed by rerunning the script.)\" \n";
	print RC_FILE "		echo \" \" \n";
	print RC_FILE "		echo \"To help debug the problem, here's the output of netstat -rn\" \n";
	print RC_FILE "		\$NETSTAT -rn\n";
	print RC_FILE "		#determine inside network, assume is /24 \" \n";
	print RC_FILE "		INSIDE_NETWORK=\${SERVER_DEFAULT_GW%.*}\n";
	print RC_FILE "		#echo \"INSIDE_NETWORK=\$INSIDE_NETWORK\"\n";
	print RC_FILE "		#netstat -rn | tail -n +3 | cut -c0-15 | grep -v \"0.0.0.0\"| grep -v \"127.0.0.0\" | grep -v -c \"\$INSIDE_NETWORK\" \n";
	print RC_FILE "		#NUMBER_OUTSIDE_ROUTES=`\$NETSTAT -rn | tail -n +3 | cut -c0-15 | grep -v \"0.0.0.0\"| grep -v \"127.0.0.0\" | grep -v -c \"\$INSIDE_NETWORK\"` \n";
	print RC_FILE "		OUTSIDE_ROUTES=`netstat -rn | tail -n +3 | cut -c0-15 | grep -v \"\$INSIDE_NETWORK\" |grep -v \"0.0.0.0\"| grep -v \"127.0.0.0\" | xargs` \n";
	print RC_FILE "		#echo \"OUTSIDE_ROUTES=\$OUTSIDE_ROUTES\" \n";
	print RC_FILE "		for OUTSIDE_ROUTE in \$OUTSIDE_ROUTES \n";
	print RC_FILE "		do\n";
	print RC_FILE "			echo \"a route to the outside world is \" \n";
	print RC_FILE "			#\$NETSTAT -rn | grep \"^192.168.2\" \n";
	print RC_FILE "			\$NETSTAT -rn | grep \"^\$OUTSIDE_ROUTE\" \n";
	print RC_FILE "			echo \"please delete it if it routes to the client and then rerun this script. \" \n";
	print RC_FILE "		done \n";
	print RC_FILE "		echo \" \" \n";
	print RC_FILE "	fi \n";
	print RC_FILE " \n";
	}
			
sub print_open_realserver
	{
	print RC_FILE "setup_realserver() {\n";
	print RC_FILE "	echo \"LVS realserver type \$LVS_TYPE \"\n";
	print RC_FILE "	#don't bother removing device with VIP if it's going to be the replacement \n";
	print RC_FILE "	NEW_VIP_DEVICE=\$SERVER_VIP_DEVICE \n";
	print RC_FILE "	remove_vip \n";
	print RC_FILE "	NEW_VIP_DEVICE=\"\" \n";
	print RC_FILE "	echo \" \"\n";
	print RC_FILE "	check_for_DIIP \n";
	print RC_FILE "	check_for_vip_on_director \n"; 

	if ($lvs_type =~ /tun|dr/)
		{
	#	print RC_FILE "	echo \"checking for VIP on realserver and/or director by pinging \" \n";
	#	print RC_FILE "	echo -n \"checking for VIP on server - \"\n";
	#	print RC_FILE "	if [ `\$IFCONFIG \$SERVER_VIP_DEVICE| grep -c \"\$VIP \"` = \"0\" ] \n";
	#	print RC_FILE "	then \n";
	#	print RC_FILE "	{ \n";
	#	print RC_FILE "		echo \"not found, good.\"\n";
	#	print RC_FILE "		echo \"When we find the VIP, it will be on director. \"\n";
	#	print RC_FILE "		echo -n \"pinging VIP (hopefully on director): \"\n";		
	#	print RC_FILE "		\$FPING \$VIP \n";	
	#	print RC_FILE "		if [ \"\$\?\" != \"0\" ] \n";
	#	print RC_FILE "		then \n";
	#	print RC_FILE "		{ \n";		
	#	if ( $vdevice eq "TP" )
	#		{
	#		print RC_FILE "			echo \"OK: director is accepting packets by TP - expect \$VIP is not pingable\" \n";		
	#		}
	#	else
	#		{
	#		print RC_FILE "			echo \"\$VIP not found, presumably director has not been set up yet\"\n";
	#		print RC_FILE "			echo \"rerun this script after setting up the director if you want all checks to be positive\"\n";
	#		}
	#	print RC_FILE "		} \n";
	#	print RC_FILE "		else \n";
	#	print RC_FILE "		{ \n";
	#	print RC_FILE "			echo \"\$VIP found, good. It's not on this server, assume it's on the director. \"\n";
	#	print RC_FILE "		} \n";
	#	print RC_FILE "		fi \n";
	#	print RC_FILE "	} \n";
	#	print RC_FILE "	else \n";
	#	print RC_FILE "	{ \n";
	#	print RC_FILE "		echo \"already on server. \"\n";
	#	print RC_FILE "		echo \"Presumably this script has been run previously on this machine. \"\n";
	#	print RC_FILE "		echo \"Won't be able to check connection to director. \"\n";
	#	print RC_FILE "		echo \"To be sure, you can delete (down) all ifconfig entries containing IP \$VIP and run this script again. \"\n";
	#	print RC_FILE "		echo \"\"\n";
	#	print RC_FILE "	} \n";
	#	print RC_FILE "	fi \n";
	#	print RC_FILE "	\n";
		install_realserver_vip_device;
		}
	else 
		{
		#lvs_type eq "nat"
		print RC_FILE "	#don't bother removing device with VIP if it's going to be the replacement \n";
		print RC_FILE "	NEW_VIP_DEVICE=\$SERVER_VIP_DEVICE \n";
		print RC_FILE "	remove_vip \n";
		print RC_FILE "	NEW_VIP_DEVICE=\"\" \n";
		check_default_route_nat
		}
	install_realserver_gw;

	}#sub print_open_realserver

sub print_close_setup_realserver
	{
	#close setup_realserver();
	print RC_FILE "	} #setup_realserver \n";
	print RC_FILE "\n";
	}
	

sub store_redirect_line
	{
	my (@redirect_line);	#for rc.horms, stores hostname, service, port, proto for TP
				#need new copy for each iteration
	#generate data for TP
	($server_name,$server_ip) = parse_hostname($server_ip);
	#push (@redirect_line, $server_ip);
	push (@redirect_line, $server_name);
	push (@redirect_line, $service_name);
	push (@redirect_line, $protocol);
	#print "store_redirect_line: service data for TP \n";
	#for $element (@redirect_line)
	#	{
	#	print "$element ";
	#	}
	#print "\n";
	push (@redirect_lines, \@redirect_line);	#store reference to @redirect_line
	#foreach $line_reference (@redirect_lines)
	#	{
	#	foreach $element (@$line_reference) { print "$element "; } print "\n";
	#	}
	#print "store_redirec_line: end \n";
	}

sub print_TP_line
	{
	#print "print_TP_line: start \n";
	#foreach $line_reference (@redirect_lines)
	#	{
	#	foreach $element (@$line_reference) { print "$element "; } print "\n";
	#	}
	#print "print_TP_line: \n";
	print RC_FILE " 	echo \"installing redirect rules\" \n";
	foreach $line_reference (@redirect_lines)
		{
		($server_name, $service_name, $protocol) = @$line_reference; 
		($server_name,$server_ip) = parse_hostname($server_name);
		# print "print_TP_line: $server_ip, $service_name, $protocol \n";
		# note blank after $server_ip. 
		# This is so that 192.168.1.1 doesn't grep match 192.168.1.11
		print RC_FILE "	if [ \`\$IFCONFIG -a | grep -c \"$server_ip \"\` = \"1\" ]\n"; 
		print RC_FILE "	then \n";
		print RC_FILE " 		echo \"redirecting \${VIP}:$service_name:$protocol 	to local:$service_name 	on $server_ip\" \n";
		print RC_FILE " 		\$IPCHAINS -A input -j REDIRECT $service_name -d \$VIP $service_name -p $protocol \n";
		print RC_FILE "	fi \n";
		#print RC_FILE "	\n";
		}
	#print RC_FILE "	\n";
	#print "\n";
	}

sub check_conf_file_input_line
	{
	#empty @params (should have been read out by read_conf_file
	foreach $element (@params)
		{
		print "error: conf_file syntax error - ";
		print "param array is not empty, contains $element \n";
		print "in line $line \n";
		shift @params;
		}
	} #check_conf_file_input_line

sub install_print_redirect_services
	{
	print RC_FILE  "print_redirect_services() {\n";
		print_TP_line;
	print RC_FILE  "	}\n";
	print RC_FILE  "\n";
	} #install_print_redirect_services

sub store_service_line
	{
	#the service line is a (possibly long) string of somewhat arbitrary syntax
	#which may specify several nodes and may (or may not) have weight(s) for nodes
	my (@service_params);	#since are using references, need a new array for each iteration
	my $counter=0;
	my $i;
	foreach $element (@params)
		{
		#print "$element \n";
		push (@service_params, $element);
		$counter++;
		}
	for ( $i =0; $i < $counter; $i++) { shift @params ;}
	push (@service_lines, \@service_params);
	#foreach $line_reference (@service_lines)
	#	{
	#	foreach $element (@$line_reference) { print "$element "; } print "\n";
	#	}
	#print "\n";
	}

sub print_service_lines
	{
	#initialise a few things, these may be set from a previous "SERVICE" line
	my $persistence='';
	my $t_switch='';
	my $timeout='';
	my $temp; #temp could be a scheduler, ppc, or an IP
	#with kernel 2.2.13, SERVICE lines do not have to specify a scheduler if ppc is used
	#no scheduling (will use default, wlc), ppc with no timeout (will use default, 360sec)
	#SERVICE=t https ppc 192.168.1.1 
	#or
	#explicit scheduling (here wrr), pcc [timeout]
	#SERVICE=t https wrr ppc 192.168.1.1
	#
	#and/or explicit timeout (here 1800 sec)
	#SERVICE=t https ppc -t 1800 192.168.1.1
	#parsing this with perl is no fun, maybe I should use yacc/flex ;-/
	#
	#print "print_service_lines: "; foreach $element (@params) { print "$element "; } print "\n";
	shift @params; #@param[0] is "SERVICE"
	($proto, $service, $temp) = @params; 
	shift @params; #shift out 2 parameters
	shift @params;
	#if $temp proves to be an IP don't want to shift it 
	#shift @params;

	#parse protocol
	if ($proto eq 't'){$protocol = "tcp";} 
	elsif ($proto eq 'u'){$protocol = "udp";} 
	#else {print "error line $line_counter: unknown protocol $proto \n";}
	else {print "print_service_lines error: unknown protocol $proto \n";}

	($service_name, $port) = parse_service($service);
	#don't have a VIP with VS_NAT, instead have an $inside_ip
	#(note - in other places I've made VS_NAT appear to use a VIP, 
	#I should make sure that VS_NAT uses the VIP notation everywhere)
	#if (defined $inside_ip) {$vip = $inside_ip;} 
	set_monitor;

	#handle scheduler/persistance
	#the next params will be one or both of scheduler/persistance 
	#(or neither, in which case it's an IP)
	#the first string will be one of rr|wrr|lc|wlc|pcc|ppc
	#if we got rr|wrr|lc|wlc|pcc then scheduler is set explicitely 
	#if pcc is found it can only be with kernel =<2.2.10, 
	#although I don't check the kernel version
	my $scheduler_known = "N";
	my $scheduler_item; 
	foreach $scheduler_item (@schedulers)
		{
		if ($temp eq $scheduler_item)
			{
			#print "scheduler $temp \n";
			$scheduler_known = "Y";
			#can shift out @params now
			$scheduler = $temp;
			shift @params;
			last;
			}
		#print "$scheduler_item \n";
		} # $scheduler_item

	if ($scheduler_known eq "Y")
		{
		#print "scheduler $scheduler found \n";
		}
	else
		{
		#print "scheduler not explicitely set \n";
		if ($temp eq "ppc")
			{
			#print "persistence found, set default scheduler, ";
			$persistence="ppc";
			#ppc isn't a scheduler, 
			#zero out $scheduler, so can assemble ipvsadm command below
			$scheduler = '';
			#can shift out @params now
			shift @params;
			#is timeout specified
			if ($params[0] eq "-t")
				{
				#non-default timeout, pop 2 strings 
				($t_switch, $timeout) = @params;
				shift @params; #shift out the 2 parameters
				shift @params;
				#print "persistence timeout $timeout \n";
				}
			else
				{
				#print "persistence timeout not specified, will use default (360 sec) \n";
				}
			} 
		else
			{
			#old way was to exit here with an error
			#print "error: scheduler $scheduler is not one of rr|wrr|lc|wlc|pcc|pcc \n";
			#exit 1;

			#now setup default scheduler
			# $temp should be an IP
			$scheduler = $default_scheduler;
			#print "next parameter $temp is an IP, scheduler not specified, use default $scheduler\n";
			} # $temp eq "-t"
		} # $scheduler_known

	#have handled cases where a scheduler or persistence is specified
	#now handle case when both are specified
	#will have to look for next arguement(s)
	#if persistence is not set, it will be the IP of the realserver handling the service
	#if persistence is set, it will be "pcc [-t timeout]" 

	if ($params[0] eq "ppc")
		{
		#print "persistence specified, non default scheduler set to $scheduler \n";
		$persistence = "ppc";
		shift @params;
		#params[0] will either be "-t" or an IP
		#look for a -t switch specifying a (non-default) timeout
		if ($params[0] eq "-t")
			{
			#non-default timeout, pop 2 strings 
			($t_switch, $timeout) = @params;
			shift @params; #shift out the 2 parameters
			shift @params;
			#print "persistence timeout $timeout \n";
			}
		else
			{
			#print "persistence found, timeout not specified, will use default (360 sec) \n";
			} # $params[0] eq "-t"
		}
	else
		{
		#print "next param $params[0] is an IP \n";
		} # $params[0] eq "ppc"

	#at last, setup connection

	print RC_FILE "	#\n";
	print RC_FILE "	#setup servers for $service \n";
	print RC_FILE "	#service $port uses";
	if ($scheduler eq '')
		{
		print RC_FILE " default scheduler (wlc)";
		}
	else
		{
		print RC_FILE " $scheduler scheduler";
		} # $scheduler eq ''

	if ($persistence eq "ppc")
		{
		print RC_FILE ", persistent connection";
		if ($timeout eq '')
			{
			print RC_FILE " with default timeout (360 sec) ";
			}
		else
			{
			print RC_FILE " with timeout $timeout sec ";
			} # $timeout
		} # $persistence eq 'ppc'

	print RC_FILE "\n";
	#print RC_FILE "	\$IPVSADM -A -$proto \${VIP}:$port";
	print RC_FILE "	\$IPVSADM -A -$proto \${VIP}:$service_name";

	if ($scheduler eq '')
		{
		#;
		}
	else
		{
		print RC_FILE " -s $scheduler";
		} # $scheduler eq ''

	if ($persistence eq "ppc")
		{
		print RC_FILE " -p";
		if ($timeout eq '')
			{
			#;
			}
		else
			{
			print RC_FILE " $timeout ";
			}
		} # $scheduler eq "ppc"

	print RC_FILE "\n";
		
	while ($server=$params[0])
		{
		#my ($server_port, $server_ipport, $server_service_name);
		my ($server_name);
		shift @params;
		#parse server
		#server will be ip[,weight] for tun|dr
		#server will be ip:port[,weight] for nat
		($server_ipport, $weight) = split /,/, $server;
		if ($weight eq ''){$weight = 1;}
		if ($initial_state eq "off")
			{$weight_lvsadm = 0;}
		else
			{$weight_lvsadm = $weight;}
		
		($server_ip, $server_port) = split /:/, $server_ipport;
		($server_name,$server_ip)=parse_hostname($server_ip);
		#add IP of server to @servers
		push (@servers, $server_ip);

		#accumulate the number of servers out there
		#use later as a test to see if I'm the director or the server
		#don't count 127.0.0.1, all machines have it.
		#
		unless ($server_ip eq "127.0.0.1") {$server_counter++};
		#print "server_counter $server_counter\n";
		
		if ($lvs_type =~ /tun|dr/)
			{ 
			#print "director tun|dr \n";
			print RC_FILE "	#echo \"adding service $service_name to realserver $server_name using connection type $lvs_type weight $weight_lvsadm\"\n";
			print RC_FILE "	\$IPVSADM -a -$proto \${VIP}:$service_name -R $server_name \$CONNECTION_TYPE -w $weight_lvsadm\n";
			print RC_FILE "	echo -n \"checking realserver $server_name reachable from director - \"\n";
			print RC_FILE "	\$FPING $server_name \n";
			if ($initial_state eq "off")
				{
				#if start with initial state off, 
				#then load ipvsadm commands above 
				#(to test that they can be loaded at run time)	
				#then delete the specific entry again
				#this way the scheduling is still on
				#if instead did ipvsadm -C, then the scheduling would be deleted too
				print RC_FILE "	echo \"director starting in off state - clearing ipvsadm entry \"\n";
				#print RC_FILE "	\$IPVSADM -d -$proto \${VIP}:$port -r ${server_ip} \n";
				#print RC_FILE "	\$IPVSADM -d -$proto \${VIP}:$service_name -r ${server_ip} \n";
				print RC_FILE "	\$IPVSADM -d -$proto \${VIP}:$service_name -r $server_name \n";
				print RC_FILE "	echo \"\"\n";
				}
			}
		elsif ($lvs_type eq "nat")
			{
			#for nat have to find port number for server too
			($server_service_name, $server_port) = parse_service($server_port);	
			print RC_FILE "	#echo \"adding service $service_name to realserver ${server_name}\"\n";
			print RC_FILE "	\$IPVSADM -a -$proto \${VIP}:$service_name -r ${server_name}:${server_port} \$CONNECTION_TYPE -w $weight_lvsadm\n";
			print RC_FILE "	echo -n \"checking if server $server_name is reachable from director: \"\n";
			#print RC_FILE "	\$FPING $server_ip \n";
			print RC_FILE "	\$FPING $server_name \n";
			if ($initial_state eq "off")
				{
				#if start with initial state off, 
				#then load ipvsadm commands above 
				#(to test that they can be loaded at run time)	
				#then delete the specific entry again
				#this way the scheduling is still on
				#if instead did ipvsadm -C, then the scheduling would be deleted too
				print RC_FILE "	echo \"director starting in off state - clearing ipvsadm ientry \"\n";
				#print RC_FILE "	\$IPVSADM -d -$proto \${VIP}:$port -r ${server_ip}:${server_port} \n";
				#print RC_FILE "	\$IPVSADM -d -$proto \${VIP}:$service_name -r ${server_ip}:${server_port} \n";
				print RC_FILE "	\$IPVSADM -d -$proto \${VIP}:$service_name -r ${server_name}:${server_port} \n";
				print RC_FILE "	echo \"\"\n";
				}
			}
		else
			{
			print "error: unknown LVS_type $lvs_type \n";
			} # $lvs_type

	store_redirect_line;
		write_mon_hostgroup;
		} # while more servers:ports on the input line

	#print RC_FILE "\n";
	} # sub print_service_lines

sub install_print_ipvsadm_lines
	{
	print RC_FILE  "print_ipvsadm_lines() {\n";
	foreach $line_reference (@service_lines)
		{
		#print "main: ";
		foreach $element (@$line_reference)
			{
			#print "$element ";
			push (@params, $element);
			}
		#print "\n";
		print_service_lines;
		}
	print RC_FILE  "	}\n";
	print RC_FILE  "\n";
	} #install_print_ipvsadm_lines

sub do_for_director_or_realserver
	{
	print RC_FILE "#Determine if I'm a realserver, director (or fail)\n";
	print RC_FILE "#Am I a server?\n";
	print RC_FILE "#The IPs listed by ifconfig are grep'ed against the list of server IPs.\n";
	print RC_FILE "#If there are no matches (== 0), then I'm not a server.\n";
	print RC_FILE "#(Note the blank after the IP.\n"; 
	print RC_FILE "#Without the blank, \"192.168.1.1\" matches \"192.168.1.1x\", but \"192.168.1.1 \" will not.\n";
	print RC_FILE "#a similar problem exists at the front of the IP, but there's no easy way to handle it.\n";
	print RC_FILE "#In linux the previous char is a blank or \'\', in solaris it's \'#\' \n";
	print RC_FILE "#If I'm not a server, then maybe I'm the director. If I'm the director, then ipvsadm will be on this machine.\n";
	print RC_FILE "#If this fails, give up, exit and cry for help\n";
	print RC_FILE "\n";
	print RC_FILE "if \\\n";
	while ($server_ip = $servers[0])	#$erver_ip may be a $server_name
		{
		($server_name,$server_ip) = parse_hostname($server_ip);
		#loop through IPs of known realservers comparing them to the IP of this machine.
		shift @servers;
		#don't test for 127.0.0.1, all machines have this, and they weren't counted with $server_counter
		if ($server_ip eq "127.0.0.1") {next;}
		$server_counter--;
		#print "server_counter $server_counter $server_ip\n";
		#print RC_FILE "[ `\$IFCONFIG $snetdevice | grep -c \"[ #]$server_ip \"` = \"0\" ] ";
		print RC_FILE "[ `\$IFCONFIG $snetdevice | grep -c \"$server_ip \"` = \"0\" ] ";
		#are we at the last server?
		if ($server_counter > 0) 
			{
			#not last server
			print RC_FILE " && \\\n";
			}
	#	else	
	#		{
	#		#last server
	#		print RC_FILE " \n";
	#		}
		}
	#print RC_FILE "if [ `\$IFCONFIG $snetdevice | grep -c \"192.168.1.8 \"` = \"0\" ] \n";
	print RC_FILE "\n";
	print RC_FILE "then \n";
	print RC_FILE "	{\n";
	print RC_FILE "	#used to test for ipvsadm \n"; 
	print RC_FILE "	#I'm the director, then \$IPVSADM should be on this machine\n";
	print RC_FILE "	#if [ -x \$IPVSADM ] \n";
	print RC_FILE "	#now test for DIIP \n"; 
	print RC_FILE " #note blank after DIRECTOR_INSIDEIP \n";
	print RC_FILE "	if [ `\$IFCONFIG -a | grep -c \"\$DIRECTOR_INSIDEIP \"` = \"1\" ] \n";
	print RC_FILE "	then\n";
	print RC_FILE "		{\n";
	print RC_FILE "		setup_director\n";
	print RC_FILE "		}\n";
	print RC_FILE "	else\n";
	print RC_FILE "		{\n";
	print RC_FILE "		echo \"error: not a server, not a director\"\n";
	print RC_FILE "		echo \"I'm not a server machine - this machine doesn\'t have the IP of any of the servers\"\n";
	#print RC_FILE "		echo \"I'm not a director - I can't execute \$IPVSADM\"\n";
	print RC_FILE "		echo \"I'm not a director - I don't have the DIIP \$DIRECTOR_INSIDEIP\"\n";
	print RC_FILE "		echo \"bummer - exiting.\"\n";
	print RC_FILE "		#return error in case this script is executed by another script\n";
	print RC_FILE "		return 1\n";
	print RC_FILE "		}\n";
	print RC_FILE "	fi\n";
	print RC_FILE "	}\n";
	print RC_FILE "else \n";
	print RC_FILE "	{\n";
	print RC_FILE "	setup_realserver\n";
	print RC_FILE "	}\n";
	print RC_FILE "fi\n";
	}

sub write_main
	{
	print RC_FILE "#-----------------------------------------\n";
	print RC_FILE "#main: \n";
	print RC_FILE "$initialise_variables";
	print RC_FILE "VIP=$vip \n";
	print RC_FILE "DIRECTOR_VIP_DEVICE=$vdevice \n";
	print RC_FILE "DIRECTOR_INSIDEIP=$inside_ip \n";
	print RC_FILE "DIRECTOR_DEFAULT_GW=$director_gw \n";
	print RC_FILE "LVS_TYPE=vs-$lvs_type \n";
	print RC_FILE "HIDDEN=\"Y\" \n";
	print RC_FILE "NAT=\"-m\" \n";
	print RC_FILE "DR=\"-g\" \n";
	print RC_FILE "TUN=\"-i\" \n";
	print RC_FILE "CONNECTION_TYPE=\"$connection_parameter\" \n";
	print RC_FILE "#echo \"connection type = \$CONNECTION_TYPE\" \n";
	if ($lvs_type eq "nat")
		{
		print RC_FILE "SERVER_DEFAULT_GW=$inside_ip \n";
		print RC_FILE "SERVER_VIP_DEVICE=\"\" \n"; #ignore incase has been set
		}
	else
		{
		print RC_FILE "SERVER_DEFAULT_GW=$server_gw \n";
		print RC_FILE "SERVER_VIP_DEVICE=$rs_vip_device \n";
		}
	print RC_FILE "\n";
	print RC_FILE "#initialise incase use TP \n";
	#print RC_FILE "SERVICES=\"";
	#foreach $element (@services)
	#	{
	#	print RC_FILE "$element ";
	#	}
	#print RC_FILE "\"\n";
	#print RC_FILE "\n";

	do_for_director_or_realserver;

	print RC_FILE "\n";
	
	print RC_FILE "#----rc.lvs_${lvs_type}-------------------------------------------------------------\n";
	} # write_main

sub read_conf_file
	{
	while (<CONF_FILE>)
		{
		$line_counter++;
		chomp;
		#chuck comments (lines starting with "#')and blank lines
		next if $_ =~ /^#|^\s*$/;
		#chuck anything after a '#'
		if ($_ =~ /#/) 
			{ 
			#need non-greedy match in case have line like
			#      #    domain             53/tcp          nameserver dns  # name-domain server
			#which is a commented comment
			#you want only the stuff before the first comment symbol
			$_ =~ /(.*?)#.*/ ; 
			$line = $1;
			}
		else
			{
			$line = $_;
			}
		#any lines blank now?
		next if $line =~ /^\s*$/;
		#print "non commented line $line \n";
		#split on '=' or white space (the '=' in the conf file is for readability only)	
		#@params = split /=|\s+/;
		@params = split /=|\s+/,$line;
		#useful print for debug	
		#print "$params[0] \n";
	
		if ($params[0] eq "LVS_TYPE")
			{
			$lvs_type = '';
			shift @params;
			if (uc($params[0]) =~/TUN/) 
				{ 
				$lvs_type = "tun"; 
				$connection_parameter= '$TUN';
				}
			if (uc($params[0]) =~/NAT/) 
				{ 
				$lvs_type = "nat"; 
				$connection_parameter= '$NAT';
				}
			if (uc($params[0]) =~/DR/) 
				{ 
				$lvs_type = "dr"; 
				$connection_parameter= '$DR';
				}
			($lvs_type =~ /tun|nat|dr/) || die "unknown LVS_TYPE $lvs_type ";
			#print "LVS_TYPE: $lvs_type \n";
			#print_headers;
			shift @params;
			}
	
		if ($params[0] eq "INITIAL_STATE")
			{
			shift @params;
			if (uc($params[0]) =~/OFF/)
				{
				$initial_state = "off";
				}
			if (uc($params[0]) =~/ON/)
				{
				$initial_state = "on";
				}
			($initial_state =~ /on|off/) || die "unknown INITIAL_STATE $params[0]";
			shift @params;
			}
	
		if ($params[0] eq "VIP") 
			{
			shift @params;
			($vdevice, $vip, $vnetmask, $vbroadcast) = @params;
			#vip could be a hostname;
			($server_name,$vip) = parse_hostname($vip);
			#print "read_conf_file: vip info $vdevice $vip $vnetmask $vbroadcast \n";
			#print_vip_info;
			#open_setup_director;
			#setup_director_vip;
			shift @params;
			shift @params;
			shift @params;
			shift @params;
			}
	
		if ($params[0] eq "DIRECTOR_INSIDEIP") 
			{
			shift @params;
			($inside_device, $inside_ip, $inside_network, $inside_netmask, $inside_broadcast) = @params;
			#inside_ip could be a hostname
			($server_name,$inside_ip) = parse_hostname($inside_ip);
			shift @params;
			shift @params;
			shift @params;
			shift @params;
			shift @params;
			}

		if ($params[0] eq "DIRECTOR_DEFAULT_GW")
			{
			shift @params;
			$director_gw = $params[0];
			#director_gw could be a hostname
			($server_name, $director_gw) = parse_hostname($director_gw);
			#print "director_gw $director_gw \n";
			shift @params; #empty
			}
	
		if ($params[0] eq "SERVER_VIP_DEVICE")
			{
			shift @params;
			$rs_vip_device= $params[0];
			shift @params;
			}
	
		if ($params[0] eq "SERVER_NET_DEVICE")
			{
			shift @params;
			$snetdevice = $params[0]; #the ethernet device on the servers 
						#that has the regular ethernet connection 	
						#for the real IP (usually eth0, le0)
			shift @params;
			}

		if ($params[0] eq "SERVER_DEFAULT_GW")
			{
			shift @params;
			$server_gw = $params[0];
			#server_gw could be a hostname
			($server_name, $server_gw) = parse_hostname($server_gw);
			#print "server_gw $server_gw \n";
			shift @params;	#empty
			}
	
		if ($params[0] eq "SERVICE")
			{
			#Initially I read in a "SERVICE" line one line at a time 
			#and then wrote out the rc.lvs lines immediately.
			#Now I store all the input till EOF(conf_file) 
			#and then output it
			#
			store_service_line;
			} # $params[0]=SERVICE

		check_conf_file_input_line;
	
		} # while CONF_FILE
	}

sub check_kernel_version
	{
	#changes can be associated with the kernel, or with ipvsadm.
	#It's a bit hard to keep track of both the kernel and ipvsadm.
	#For the moment, just keep track of the kernel.
	#are we running this on a kernel version that has been tested with configure?
	$uname_r = `uname -r`;
	chomp ($uname_r);
	#print "uname_r $uname_r \n";
	#wierd names. 
	#patched kernels have names like 2.2.15pre9, 2.3.99-pre8 or 2.4.0-test1
	#I was hoping to match on the pre until the -pre and -test format came along.
	#Now I just look for non-digits.
	#originally I decided that a 2.2.15pre9 kernel should be treated as a 2.2.14
	#then 2.4.0-test came along. I can't treat it as a "-1". Will treat "0" as a special case
	#the kernel could be 2.2.15pre5 -rev it down to 2.2.14
	#test cases
	#$uname_r = "2.2.1test-9";
	#$uname_r = "2.2.1-test9";
	#$uname_r = "2.2.1";
	$uname_r =~ /(\d)\.(\d)\.(\d{1,3})[\-A-Za-z]*(\d{0,3})/ ;
	#print "kernel version $1 $2 $3 $4\n";
	if ($4 eq "")
		{
		#print "standard release \n";
		#print "kernel version $1.$2.$3\n";
		}
	else	
		{
		#print "pre or test release \n";
		#print "$1.$2.$3pre$4;\n";
		$minor_rev = $3 - 1;
		if ($minor_rev eq -1) {$minor_rev = 0;}
		print "kernel $uname_r is being treated as ";
		$uname_r = "$1.$2.$minor_rev";
		print "$uname_r\n";
		}
	#print "uname $uname_r \n";
	
	#check if this is a known $uname_r
	foreach $uname_r_item (@uname_r_ok)
		{
		if ($uname_r eq $uname_r_item)
			{
			$uname_r_known = "Y";
			last;
			}
		#print "$uname_r_item \n";
		}
	if ($uname_r_known eq "N")
		{
		print "warning: $0 has not been tested with kernel version $uname_r \n";
		}
	}	

sub sanity_check
	{
	#random list of incompatibilities in conf-file
	#exit if find any of these
	
	}

#----------------------------------------------------------
#main:

if (@ARGV == 0)
	{
	print "usage: ./configure_lvs.pl conf_file \n";
	exit;
	}

$conf_file = shift(@ARGV);
#print "conf_file: $conf_file\n";
open (CONF_FILE, "<$conf_file")|| die "Can't read $conf_file $!\n";

check_kernel_version;	#looks at kernel version on director.
			#kernel version on realserver 
			#determined in detect_realserver_os when shell script is run
read_conf_file;
sanity_check;
open_files;
print_headers;

#write a few subroutines
install_print_ipvsadm_lines;
install_print_redirect_services;

#setup director and mon 
open_setup_director;
if ($vdevice eq "TP")
	{
	#print "setting up TP for VIP director \n";
	setup_director_TP;
	}
else	{
	#print "setting up network device for VIP on director \n";
	setup_director_vip;
	}
setup_ipvsadm_table;
setup_director_masq_table;
#currently setting this up on machine known to be director
#eventually will have this already setup and will use this IP
#to deteect whether are director or not
#setup_director_insideip;

print_close_setup_director;
print_close_mon_file;

#setup realservers
print_open_realserver;
print_close_setup_realserver;

#main
write_main;

close MON_FILE;
close RC_FILE;
#make rc.lvs executable
my @cannot;
@cannot = grep {not chmod 755, $_} $rc_file;
die "$0: could not chmod @cannot\n" if @cannot;

#-----configure_lvs-2.2.15.pl---------------------------------
